/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * Singleton class to handle actions on folders for the resources explorer.
 * @private
 */
Ext.define('Ametys.plugins.explorer.applications.resources.ResourcesControllerActions.Folder', {
	singleton: true,
	
	/**
	 * @private
	 * @readonly
	 * @property {RegExp} _targetFilter The regexp that match explorer related target type.
	 */
	_targetFilter : /^(explorer-collection|resource)$/,
	
	//------------------------------------------------------//
	//					ADD FOLDER							//
	//------------------------------------------------------//
	
	/**
	 * Function called to add a new folder
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	add: function(controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (target)
		{
			Ametys.explorer.resources.actions.Folder.add(
				target.getParameters().id,
				"{{i18n PLUGINS_EXPLORER_FOLDER_HANDLE_ADD_NEWNAME}}"
			);
		}
	},
	
	//------------------------------------------------------//
	//					ADD CMIS							//
	//------------------------------------------------------//
	/**
	 * Function called to add a new CMIS folder
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	addCMIS: function(controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (target)
		{
			Ametys.data.ServerComm.callMethod({
				role: 'org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO', 
				methodName: 'isCMISCollection',
				parameters: [target.getParameters().id],
				callback: {
					handler: this._isCMIS,
					scope: this,
					arguments: {
						target: target
					}
				},
				errorMessage: "{{i18n plugin.explorer:PLUGINS_EXPLORER_CMIS_IS_ROOT_ERROR}}"
			});
		}
	},
	
	/**
	 * Callback function invoked after checked CMIS root properties
	 * @param {Boolean} isCMIS true if the current node is a CMIS collection
	 * @param {Object} args The callback arguments
	 * @private
	 */
	_isCMIS: function (isCMIS, args)
	{
		Ametys.explorer.resources.actions.Folder.addCMIS(args.target.getParameters().id, isCMIS ? 'edit' : 'new');
	},
	
	
	//------------------------------------------------------//
	//						RENAME							//
	//------------------------------------------------------//
	/**
	 * Function called to rename a folder
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	rename: function(controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (target)
		{
			// Need to retrieves explorer node to be sure to get the current name.
			var me = this,
				id = target.getParameters().id;
			
			Ametys.explorer.ExplorerNodeDAO.getExplorerNode(id, function(explorerNode) {
				
				var name = explorerNode.getProperties().name;
				
				Ametys.Msg.prompt(
					"{{i18n plugin.explorer:PLUGINS_EXPLORER_FOLDER_HANDLE_RENAME}}", 
					"{{i18n plugin.explorer:PLUGINS_EXPLORER_FOLDER_NAME}}", 
					function(btn, text) {
						if (btn == 'ok')
						{
							me._doRename(text, id);
						}
					},
					this,
					false,
					name
				);
			});
		}
	},
	
	/**
	 * Function called after validating the prompt dialog box
	 * @param {String} text Value of the input field (new folder name)
	 * @param {String} id The id og the folder to renamed
	 * @private
	 */
	_doRename: function(text, id)
	{
		Ametys.explorer.resources.actions.Folder.rename(id, text);
	},
	
	//------------------------------------------------------//
	//						REMOVE							//
	//------------------------------------------------------//
	/**
	 * Function called to delete a folder
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	remove: function(controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (target)
		{
			Ametys.explorer.resources.actions.Folder.remove(target.getParameters().id);
		}
	},
	
	//------------------------------------------------------//
	//						ARCHIVE							//
	//------------------------------------------------------//
	/**
	 * Function called to export a folder to a ZIP archive
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	archive: function (controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (target)
		{
			var name;
			if (target.getParameters().id == target.getParameters().rootId)
			{
				name = "{{i18n PLUGINS_EXPLORER_ROOT_NODE}}";
			}
			Ametys.explorer.resources.actions.Folder.archive(target.getParameters().id, name);
		}
	}
});

/**
 * Singleton class to handle actions on files for the resources explorer.
 * @private
 */
Ext.define('Ametys.plugins.explorer.applications.resources.ResourcesControllerActions.File', {
	singleton: true,
	
	/**
	 * @private
	 * @readonly
	 * @property {RegExp} _targetFilter The regexp that match explorer related target type.
	 */
	_targetFilter : /^(explorer-collection|resource)$/,
	
	/**
	 * @private
	 * @readonly
	 * @property {RegExp} _targetFileFilter The regexp that match only explorer file.
	 */
	_targetFileFilter: /^resource$/,
	
	//------------------------------------------------------//
	//						ADD								//
	//------------------------------------------------------//
	/**
	 * Function called to upload a new file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	add: function(controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (target)
		{
			Ametys.explorer.resources.actions.File.add(target.getParameters().id);
		}
	},

	//------------------------------------------------------//
	//						RENAME							//
	//------------------------------------------------------//
	/**
	 * Function called to rename a file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	rename: function(controller)
	{
		// Search for the first explorer resource
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFileFilter);
		if (!target)
		{
			return;
		}
		
		// Need to retrieves the resource node to be sure to get the current name.
		var me = this,
			id = target.getParameters().id;
		
		Ametys.explorer.ExplorerNodeDAO.getExplorerNode(id, function(resource) {
			
			var name = resource.getProperties().name;
			
			Ametys.Msg.prompt(
				"{{i18n plugin.explorer:PLUGINS_EXPLORER_FILE_HANDLE_RENAME}}", 
				"{{i18n plugin.explorer:PLUGINS_EXPLORER_FILE_NAME}}", 
				function (btn, text) {
					if (btn == 'ok')
					{
						Ametys.explorer.resources.actions.File.rename(target.getParameters().id, name, text);
					}
				},
				me, 
				false,
				name
			);
		});
	},
	
	//------------------------------------------------------//
	//					DUBLIN CORE							//
	//------------------------------------------------------//
	/**
	 * Function called to edit the Dublin Core metadata of a file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	editDublinCore: function(controller)
	{
		// Search for the first explorer resource
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFileFilter);
		if (target)
		{
			Ametys.explorer.resources.actions.File.editDublinCore(target.getParameters().id);
		}
	},
	
	//------------------------------------------------------//
	//					REMOVE								//
	//------------------------------------------------------//
	/**
	 * Function called to delete a file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	remove: function(controller)
	{
		// Search for the  explorer resources
		var targets = Ametys.message.MessageTargetHelper.findTargets(controller.getMatchingTargets(), this._targetFileFilter);
		if (!targets[0])
		{
			return;
		}
		
		var ids = [];
		Ext.Array.forEach(targets, function(target) {
			ids.push(target.getParameters().id);
		});
		
		var me = this;
		Ametys.explorer.resources.actions.File.remove(targets[0].getParameters().parentId, ids);
	},
	
	//------------------------------------------------------//
	//					DOWNLOAD							//
	//------------------------------------------------------//
	/**
	 * Function called to download a file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	download: function(controller)
	{
		// Search for the  explorer resource
		var targets = Ametys.message.MessageTargetHelper.findTargets(controller.getMatchingTargets(), this._targetFileFilter);
		if (!targets[0])
		{
			return;
		}
		
		if (targets.length == 1)
		{
			Ametys.explorer.resources.actions.File.download(targets[0].getParameters().id);
		}
		else
		{
			var ids = [];
			
			Ext.Array.forEach(targets, function(target) {
				ids.push(target.getParameters().id);
			});
			
			Ametys.explorer.resources.actions.File.exportZIP(targets[0].getParameters().parentId, ids);
		}
		
	},
	
	//------------------------------------------------------//
	//					HISTORY								//
	//------------------------------------------------------//
	/**
	 * Function called to view the old versions of a file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	viewHistory: function(controller)
	{
		// Search for the first explorer resource
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFileFilter);
		if (!target)
		{
			return;
		}
		
		Ametys.explorer.resources.actions.File.showHistory(target.getParameters().id, target.getParameters().parentId, true);
	},
	
	//------------------------------------------------------//
	//					SLIDE SHOW							//
	//------------------------------------------------------//
	/**
	 * Launch a slide show on images stored by the controller
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	slideShow: function (controller)
	{
		// Search for the first explorer resource
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (!target)
		{
			return;
		}
		
		Ametys.explorer.resources.actions.File.slideshow(controller._images);
	},
	
	//------------------------------------------------------//
	//						SEARCH							//
	//------------------------------------------------------//
	
	openSearch: function(controller)
	{
		// Search for the first explorer target (explorer node).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (!target)
		{
			return;
		}
		
		var toolId = controller.getInitialConfig('search-tool-id') || 'uitool-explorer-search';
		
		var targetParams = target.getParameters();
		Ametys.tool.ToolsManager.openTool(toolId, {
			id: targetParams.id,
			applicationId: targetParams.applicationId,
			path: targetParams.path,
			rootId: targetParams.rootId
		});
	},
	
	//------------------------------------------------------//
	//				COPY / CUT / PASTE						//
	//------------------------------------------------------//
	/**
	 * Copy the current selected files
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	copy: function(controller)
	{
		// Search for the  explorer resource
		var targets = Ametys.message.MessageTargetHelper.findTargets(controller.getMatchingTargets(), this._targetFileFilter);
		if (targets.length == 0)
		{
			return;
		}
		
		var ids = [];
		Ext.Array.forEach(targets, function(target) {
			ids.push(target.getParameters().id);
		});
		
		Ametys.explorer.resources.actions.File.copy(targets[0].getParameters().parentId, ids);
	},
	
	/**
	 * Cut the current selected files
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	cut: function(controller)
	{
		// Search for the  explorer resource
		var targets = Ametys.message.MessageTargetHelper.findTargets(controller.getMatchingTargets(), this._targetFileFilter);
		if (targets.length == 0)
		{
			return;
		}
		
		var ids = [];
		Ext.Array.forEach(targets, function(target) {
			ids.push(target.getParameters().id);
		});
		
		Ametys.explorer.resources.actions.File.cut(targets[0].getParameters().parentId, ids);
	},
	
	/**
	 * Paste the cut or copied file
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	paste: function(controller)
	{
		// Search for the first explorer target (explorer node or resource).
		var target = Ametys.message.MessageTargetHelper.findTarget(controller.getMatchingTargets(), this._targetFilter);
		if (!target)
		{
			return;
		}
		
		Ametys.explorer.resources.actions.File.paste(target.getParameters().id);
	}
});

/**
 * Singleton class to handle actions on folders for the resources explorer.
 * @private
 */
Ext.define('Ametys.plugins.explorer.applications.resources.ResourcesControllerActions.Views', {
	singleton: true,
	
	/**
	 * Change the view of the resource collection tool
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
	 */
	changeView: function(controller)
	{
		var tool = Ametys.tool.ToolsManager.getFocusedTool(),
			view = controller.getInitialConfig("resource-collection-view");
		
		if (tool != null && (view == 'details' || view == 'images'))
		{
			if (tool.getParams().view != view)
			{
				tool.changeView(view)
			}
		}
	},
});
