/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class provides a view for the file.
 * This is the 'thumbnail' view, which is based on a dataview.
 * @private
 */
Ext.define('Ametys.plugins.explorer.view.ImageThumbnailViewer', {
	extend: 'Ext.view.View',
	
	/**
	 * @property {String} _applicationId The application id
	 * @private
	 */
	
	/**
	 * @cfg
	 * @inheritdoc
	 */
	cls: 'images-view',
	/**
	 * @cfg
	 * @inheritdoc
	 */
	overItemCls: 'x-item-over',
	/**
	 * @cfg
	 * @inheritdoc
	 */
	itemSelector: 'div.thumb-wrap',
	
	multiSelect: true,
	trackOver: true,
	
	constructor: function(config)
	{
		// plugins
		config.plugins = config.plugins || [];
		
		var labelEditorPlugin = Ext.create('Ext.ux.DataView.LabelEditor', {
			dataIndex: 'name',
			
			listeners: {
				beforestartedit: {fn: this._onBeforeStartEdit, scope: this},
				beforecomplete: {fn: this._onBeforeComplete, scope: this}
			},
			
			onSave: Ext.bind(this._onEditorSave, this)
		});
		config.plugins.push(labelEditorPlugin);
		
		// tpl
		config.tpl = config.tpl || Ext.create('Ext.XTemplate', [
			'<tpl for=".">',
				'<div class="thumb-wrap" id="{name:stripTags}">',
					'<div class="thumb"><img src="{thumbnailPath}" title="{name:htmlEncode}"></div>',
					'<span class="x-editable">{shortName:htmlEncode}</span>',
				'</div>',
			'</tpl>',
			'<div class="x-clear"></div>'
		]);
		
		this.callParent(arguments);
	},
	
	/**
	 * Set the application id for this view
	 * @param {String} applicationId the application id
	 */
	setApplicationId: function(applicationId)
	{
		this._applicationId = applicationId;
	},
	
	/**
	 * @private
	 * Listen to the 'beforestartedit' event of the editor.
	 * Handle our internal logic for the edit process.
	 * @param {Ext.Editor} editor The label editor
	 * @param {Ext.dom.Element} boundEl The underlying element bound to this editor 
	 * See {@link Ext.Editor#event-beforestartedit}
	 */
	_onBeforeStartEdit: function(editor, boundEl)
	{
		var item = this.findItemByChild(boundEl),
			record = this.store.getAt(this.indexOf(item)),
			path = record.get('path');
	
		if (record.get('isModifiable') !== true)
		{
			return false;
		}
	
		// FIXME can rename node could be async...
		if (!Ametys.explorer.ExplorerNodeDAO.canRenameFile(path))
		{
			return false;
		}
	},
	
	/**
	 * @private
	 * Listen to the 'beforecomplete' event of the editor.
	 * Handle our internal logic for the edit process.
	 * @param {Ext.Editor} editor The label editor
	 * @param {String} value The current field value
	 * @param {String} startValue The original field value
	 * See {@link Ext.Editor#event-beforecomplete}
	 */
	_onBeforeComplete: function(editor, value, startValue)
	{
		// IE < 9 does not support trim. Using the Ext function.
		if (Ext.String.trim(value) == Ext.String.trim(startValue))
		{
			editor.cancelEdit();
			return false;
		}
	},
	
	/**
	 * @private
	 * End of the edition process, perform the resource renaming
	 * @param {Ext.Editor} ed The label editor
	 * @param {String} value The current field value
	 * @param {String} startValue The original field value
	 */
	_onEditorSave: function(ed, value, startValue)
	{
		var record = ed.activeRecord;
		if (record && value && startValue)
		{
			var name = Ext.String.trim(value),
				oldName = Ext.String.trim(startValue);
			
			if (name != oldName)
			{
				Ametys.explorer.resources.actions.File.rename(
					record.getId(),
					oldName,
					name
				);
			}
		}
	}
});
