/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.generators.ResourcesExplorerGenerator;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;

/**
 * Generates a subtree of the resources explorer.
 */
public class ExplorerNodeGenerator extends ServiceableGenerator
{
    /** The resolver for ametys object */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _resolver = (AmetysObjectResolver) sManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = request.getParameter("node");
        ExplorerNode node = _resolver.resolveById(id);
        
        _saxExplorerNode(node);
    }
    
    private void _saxExplorerNode (ExplorerNode node) throws SAXException
    {
        contentHandler.startDocument();
        
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", node.getId());
        
        XMLUtils.startElement(contentHandler, "Nodes", atts);
        
        if (node instanceof TraversableAmetysObject)
        {
            TraversableAmetysObject traversableObject = (TraversableAmetysObject) node;
            AmetysObjectIterable<AmetysObject> children = traversableObject.getChildren();
            
            for (AmetysObject child : children)
            {
                if (child instanceof ExplorerNode)
                {
                    saxExplorerNode((ExplorerNode) child);
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, "Nodes");
        
        contentHandler.endDocument();
    }
    
    /**
     * Sax a {@link ExplorerNode}
     * @param node The node to sax
     * @throws SAXException If an error occurred while saxing
     */
    protected void saxExplorerNode (ExplorerNode node) throws SAXException
    {
        AttributesImpl childAtts = new AttributesImpl();
        childAtts.addCDATAAttribute("id", node.getId());
        childAtts.addCDATAAttribute("name", node.getName());
        childAtts.addCDATAAttribute("iconCls", node.getIconCls());
        childAtts.addCDATAAttribute("applicationId", node.getApplicationId());
        
        String relPath = node.getExplorerPath();
        childAtts.addCDATAAttribute("path", relPath.startsWith("/") ? relPath.substring(1) : relPath);
        childAtts.addCDATAAttribute("type", ResourcesExplorerGenerator.RESOURCE_COLLECTION);
        
        boolean hasResources = false;
        if (node instanceof ResourceCollection)
        {
            hasResources = ((ResourceCollection) node).hasChildResources();
        }
        boolean hasChildNodes = node.hasChildExplorerNodes();
        
        if (hasChildNodes)
        {
            childAtts.addCDATAAttribute("hasChildNodes", "true");
        }
        
        if (hasResources)
        {
            childAtts.addCDATAAttribute("hasResources", "true");
        }
        
        if (node instanceof ModifiableAmetysObject)
        {
            childAtts.addCDATAAttribute("isModifiable", "true");
        }
        
        if (node instanceof ModifiableExplorerNode)
        {
            childAtts.addCDATAAttribute("canCreateChild", "true");
        }
        
        XMLUtils.createElement(contentHandler, "Node", childAtts);
    }
}
