/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.cmis;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;

import org.apache.chemistry.opencmis.client.api.Document;
import org.apache.chemistry.opencmis.client.api.Folder;
import org.apache.chemistry.opencmis.commons.data.ContentStream;
import org.apache.commons.lang3.Strings;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * Implementation of an {@link Resource}, backed by a server CMIS.<br>
 */
public class CMISResource implements Resource
{
    private final Document _cmisDocument;
    private final CMISRootResourcesCollection _root;
    private AmetysObject _parent;
    
    /**
     * Creates a {@link CMISResource}
     * @param document The CMIS document
     * @param root The root of the virtual tree
     * @param parent the parent {@link AmetysObject} if known
     */
    public CMISResource(Document document, CMISRootResourcesCollection root, AmetysObject parent)
    {
        _cmisDocument = document;
        _root = root;
        _parent = parent;
    }

    /**
     * Retrieves the {@link CMISRootResourcesCollection}
     * @return the {@link CMISRootResourcesCollection}
     */
    public CMISRootResourcesCollection getCmisRoot ()
    {
        return _root;
    }
    
    /**
     * Retrieves the {@link Folder}
     * @return the {@link Folder}
     */
    public Document getCmisDocument ()
    {
        return _cmisDocument;
    }
    
    @Override
    public String getId() throws AmetysRepositoryException
    {
        return getCmisRoot().getId() + "/" + getCmisDocument().getId();
    }

    @Override
    public String getName()
    {
        String fileName = getCmisDocument().getName();

        if (fileName == null)
        {
            fileName = getCmisDocument().getContentStreamFileName();
        }
        if (fileName == null)
        {
            ContentStream cs = getCmisDocument().getContentStream();
            if (cs != null)
            {
                fileName = cs.getFileName();
            }
            else
            {
                throw new AmetysRepositoryException("contentStream is empty on CMIS Document '" + getCmisDocument().getId() + "'");
            }

            if (fileName == null)
            {
                throw new AmetysRepositoryException("fileName is null on CMIS Document '" + getCmisDocument().getId() + "'");
            }
        }
        return fileName;
    }

    @Override
    @SuppressWarnings("unchecked")
    public AmetysObject getParent() throws AmetysRepositoryException
    {
        if (_parent != null)
        {
            return _parent;
        }
        
        // iterate on all parents and returns the first valid parent
        String rootPath = _root.getRootFolder().getPath();
        String rootId = _root.getRootFolder().getId();
        for (Folder parent : getCmisDocument().getParents())
        {
            // check if the parent is valid by making sure that it is below the root folder
            // This is necessary to unsure that the parent is included in the mount point
            if (Strings.CS.contains(parent.getPath(), rootPath))
            {
                if (Strings.CS.equals(parent.getId(), rootId))
                {
                    _parent = getCmisRoot();
                }
                else
                {
                    _parent = new CMISResourcesCollection(parent, _root, null);
                }
                return _parent;
            }
        }
        
        throw new AmetysRepositoryException("Failed to retrieve the parent of folder " + getName() + ". This is most likely due to the fact that we couldn't find one in the CMIS mount point.");
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return getParent().getPath();
    }

    @Override
    public String getPath() throws AmetysRepositoryException
    {
        return getParentPath() + "/" + getName();
    }

    @Override
    public UserIdentity getCreator()
    {
        // FIXME We cannot convert the CMIS remote user to an Ametys user. We need to know the population. If not found, return the system user.
        // getCmisDocument().getCreatedBy();
        return UserPopulationDAO.SYSTEM_USER_IDENTITY;
    }

    @Override
    public InputStream getInputStream() throws AmetysRepositoryException
    {
        ContentStream contentStream = getCmisDocument().getContentStream();
        if (contentStream != null)
        {
            return contentStream.getStream();
        }
        return null;
    }

    @Override
    public Date getLastModified()
    {
        Calendar calendar = getCmisDocument().getLastModificationDate();
        return calendar.getTime();
    }

    @Override
    public long getLength()
    {
        return getCmisDocument().getContentStreamLength();
    }

   
    @Override
    public String getMimeType() throws AmetysRepositoryException
    {
        String mimeType = getCmisDocument().getContentStreamMimeType();
        
        return mimeType != null ? mimeType : "application/unknown";
    }

    @Override
    public String[] getKeywords() throws AmetysRepositoryException
    {
        return new String[0];
    }

    @Override
    public String getKeywordsAsString() throws AmetysRepositoryException
    {
        return "";
    }
    
    @Override
    public String getResourcePath() throws AmetysRepositoryException
    {
        return ((ExplorerNode) getParent()).getExplorerPath() + "/" + getName();
    }
    
    // Dublin Core metadata. //
    
    @Override
    public String getDCTitle() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCCreator() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String[] getDCSubject() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCDescription() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCPublisher() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCContributor() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public Date getDCDate() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCType() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCFormat() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCIdentifier() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCSource() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCLanguage() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCRelation() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCCoverage() throws AmetysRepositoryException
    {
        return null;
    }
    
    @Override
    public String getDCRights() throws AmetysRepositoryException
    {
        return null;
    }

    public Date getCreationDate() throws AmetysRepositoryException
    {
        Calendar calendar = getCmisDocument().getCreationDate();
        return calendar.getTime();
    }

    public UserIdentity getLastContributor() throws AmetysRepositoryException
    {
        return UserIdentity.stringToUserIdentity(getCmisDocument().getLastModifiedBy());
    }
}
