/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.cmis;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.chemistry.opencmis.client.api.CmisObject;
import org.apache.chemistry.opencmis.client.api.Document;
import org.apache.chemistry.opencmis.client.api.Folder;
import org.apache.chemistry.opencmis.client.api.ItemIterable;
import org.apache.chemistry.opencmis.client.api.Session;
import org.apache.chemistry.opencmis.commons.enums.BaseTypeId;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * Implementation of an {@link ExplorerNode}, backed by a CMIS server.<br>
 */
public class CMISResourcesCollection implements ResourceCollection
{
    /** Application id for resources collections. */
    public static final String APPLICATION_ID = "Ametys.plugins.explorer.applications.resources.Resources";

    private static final Logger __LOGGER = LoggerFactory.getLogger(CMISResourcesCollection.class);
    
    private final Folder _cmisFolder;
    private final CMISRootResourcesCollection _root;
    private AmetysObject _parent;
    
    /**
     * Creates a {@link CMISResourcesCollection}
     * @param folder The CMIS folder
     * @param root The root of the virtual tree
     * @param parent the parent {@link AmetysObject} if known
     */
    public CMISResourcesCollection(Folder folder, CMISRootResourcesCollection root, AmetysObject parent)
    {
        _cmisFolder = folder;
        _root = root;
        _parent = parent;
    }
    
    @Override
    public String getApplicationId()
    {
        return APPLICATION_ID;
    }

    @Override
    public String getIconCls()
    {
        return "ametysicon-folder249";
    }

    @Override
    public String getId() throws AmetysRepositoryException
    {
        return getCmisRoot().getId() + "/" + getCmisFolder().getId();
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        return getCmisFolder().getName();
    }
    
    /**
     * Retrieves the {@link CMISRootResourcesCollection}
     * @return the {@link CMISRootResourcesCollection}
     */
    public CMISRootResourcesCollection getCmisRoot ()
    {
        return _root;
    }
    
    /**
     * Retrieves the {@link Folder}
     * @return the {@link Folder}
     */
    public Folder getCmisFolder ()
    {
        return _cmisFolder;
    }

    @SuppressWarnings("unchecked")
    @Override
    public AmetysObject getParent() throws AmetysRepositoryException
    {
        if (_parent != null)
        {
            return _parent;
        }
        
        // iterate on all parents and returns the first valid parent
        String rootPath = _root.getRootFolder().getPath();
        String rootId = _root.getRootFolder().getId();
        for (Folder parent : _cmisFolder.getParents())
        {
            // check if the parent is valid by making sure that it is below the root folder
            // This is necessary to unsure that the parent is included in the mount point
            if (Strings.CS.contains(parent.getPath(), rootPath))
            {
                if (Strings.CS.equals(parent.getId(), rootId))
                {
                    _parent = getCmisRoot();
                }
                else
                {
                    _parent = new CMISResourcesCollection(parent, _root, null);
                }
                return _parent;
            }
        }
        
        throw new AmetysRepositoryException("Failed to retrieve the parent of folder " + getName() + ". This is most likely due to the fact that we couldn't find one in the CMIS mount point.");
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return getParent().getPath();
    }

    @Override
    public String getPath() throws AmetysRepositoryException
    {
        return getParentPath() + "/" + getName();
    }
    
    @Override
    @SuppressWarnings("unchecked")
    public AmetysObject getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        Session session = getCmisRoot().getSession();
        if (session == null)
        {
            throw new UnknownAmetysObjectException("Failed to connect to CMIS server");
        }
        
        CmisObject child = session.getObjectByPath(path);
        BaseTypeId baseTypeId = child.getBaseType().getBaseTypeId();
        
        if (baseTypeId.equals(BaseTypeId.CMIS_FOLDER))
        {
            return new CMISResourcesCollection((Folder) child, _root, this);
        }
        if (baseTypeId.equals(BaseTypeId.CMIS_DOCUMENT))
        {
            Document cmisDoc = (Document) child;
            // Check if CMIS document has content, if not ignore it
            if (cmisDoc.getContentStream() != null)
            {
                return new CMISResource((Document) child, _root, this);
            }
            throw new UnknownAmetysObjectException("The CMIS document " + path + " has no content");
        }
        else
        {
            throw new UnknownAmetysObjectException("Unhandled CMIS type '" + baseTypeId + "', cannot get child at path " + path);
        }
    }

    
    @Override
    public CollectionIterable<AmetysObject> getChildren() throws AmetysRepositoryException
    {
        Collection<AmetysObject> aoChildren = new ArrayList<>();
        
        ItemIterable<CmisObject> children = getCmisFolder().getChildren();
        for (CmisObject child : children)
        {
            BaseTypeId baseTypeId = child.getBaseTypeId();
            if (baseTypeId.equals(BaseTypeId.CMIS_FOLDER))
            {
                aoChildren.add(new CMISResourcesCollection((Folder) child, _root, this));
            }
            else if (baseTypeId.equals(BaseTypeId.CMIS_DOCUMENT))
            {
                Document cmisDoc = (Document) child;
                
                // Check if CMIS document has content, if not ignore it
                if (StringUtils.isNotEmpty(cmisDoc.getContentStreamFileName()))
                {
                    aoChildren.add(new CMISResource(cmisDoc, _root, this));
                }
            }
            else
            {
                __LOGGER.warn("Unhandled CMIS type {}. It will be ignored.", baseTypeId);
            }
        }

        return new CollectionIterable<>(aoChildren);
    }

    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        ItemIterable<CmisObject> children = _cmisFolder.getChildren();
        for (CmisObject child : children)
        {
            if (child.getName().equals(name))
            {
                return true;
            }
        }
        
        return false;
    }
    
    public boolean hasChildResources() throws AmetysRepositoryException
    {
        // we don't actually know if there are children or not,
        // but it's an optimization to don't make another CMIS request
        return true;
    }
    
    public boolean hasChildExplorerNodes() throws AmetysRepositoryException
    {
        // we don't actually know if there are children or not,
        // but it's an optimization to don't make another CMIS request
        return true;
    }
    
    @Override
    public String getExplorerPath()
    {
        AmetysObject parent = getParent();
        
        if (parent instanceof ExplorerNode)
        {
            return ((ExplorerNode) parent).getExplorerPath() + "/" + getName();
        }
        else
        {
            return "";
        }
    }
    
    @Override
    public String getResourcePath() throws AmetysRepositoryException
    {
        return getExplorerPath();
    }
    
    @Override
    public String getDescription()
    {
        return null;
    }
}
