/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.cmis;

import java.util.ArrayList;
import java.util.Collection;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.nodetype.ConstraintViolationException;

import org.apache.chemistry.opencmis.client.api.CmisObject;
import org.apache.chemistry.opencmis.client.api.Document;
import org.apache.chemistry.opencmis.client.api.Folder;
import org.apache.chemistry.opencmis.client.api.ItemIterable;
import org.apache.chemistry.opencmis.client.api.Session;
import org.apache.chemistry.opencmis.commons.enums.BaseTypeId;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AbstractAmetysObject;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.SimpleAmetysObjectFactory;
import org.ametys.plugins.repository.metadata.ModifiableCompositeMetadata;
import org.ametys.plugins.repository.metadata.jcr.JCRCompositeMetadata;

/**
 * {@link AmetysObject} implementing the root of {@link CMISResourcesCollection}s
 */
public class CMISRootResourcesCollection extends AbstractAmetysObject implements JCRAmetysObject, ResourceCollection, ModifiableModelLessDataAwareAmetysObject
{
    /** application id for resources collections. */
    public static final String APPLICATION_ID = "Ametys.plugins.explorer.applications.resources.Resources";
   
    /** Metadata for repository id */
    public static final String DATA_REPOSITORY_ID = "repositoryId";
    /** Metadata for repository url */
    public static final String DATA_REPOSITORY_URL = "repositoryUrl";
    /** Attribute for repository root path */
    public static final String DATA_MOUNT_POINT = "mountPoint";
    /** Metadata for user login */
    public static final String DATA_USER = "user";
    /** Metadata for user password */
    public static final String DATA_PASSWORD = "password";
   
    private static final Logger __LOGGER = LoggerFactory.getLogger(CMISResourcesCollection.class);
    
    /** The corresponding {@link SimpleAmetysObjectFactory} */
    private final CMISTreeFactory _factory;
    
    private final Node _node;
    
    private String _name;
    private String _parentPath;
    
    private Session _session;
    private Folder _root;
    
    /**
     * Creates a {@link CMISRootResourcesCollection}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the CMISRootResourcesCollectionFactory which created this AmetysObject
     */
    public CMISRootResourcesCollection(Node node, String parentPath, CMISTreeFactory factory)
    {
        _node = node;
        _parentPath = parentPath;
        _factory = factory;
        
        try
        {
            _name = _node.getName();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get node name", e);
        }
    }
    
    void connect(Session session, Folder root)
    {
        _session = session;
        _root = root;
    }
    
    Session getSession()
    {
        return _session;
    }

    Folder getRootFolder()
    {
        return _root;
    }

    @SuppressWarnings("unchecked")
    @Override
    public AmetysObject getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (_session == null)
        {
            throw new UnknownAmetysObjectException("Failed to connect to CMIS server");
        }
        
        String mountPoint = StringUtils.defaultIfBlank(getMountPoint(), "/");
        // ensure mount point start with "/" (Chemistry restriction)
        if (!mountPoint.startsWith("/"))
        {
            mountPoint = "/" + mountPoint;
        }
        // add trailing slash if needed
        if (!mountPoint.endsWith("/"))
        {
            mountPoint = mountPoint + "/";
        }
        
        String remotePath = mountPoint + path;
        // ensure path does not ends with "/" (Chemistry restriction)
        if (path.endsWith("/") && remotePath.length() != 1)
        {
            remotePath = remotePath.substring(0, remotePath.length() - 1);
        }
        
        CmisObject entry = _session.getObjectByPath(remotePath);
        CmisObject object = _session.getObject(entry);
        
        BaseTypeId baseTypeId = object.getBaseType().getBaseTypeId();
        
        if (baseTypeId.equals(BaseTypeId.CMIS_FOLDER))
        {
            return new CMISResourcesCollection((Folder) object, this, null);
        }
        else if (baseTypeId.equals(BaseTypeId.CMIS_DOCUMENT))
        {
            return new CMISResource((Document) object, this, null);
        }
        else
        {
            throw new UnknownAmetysObjectException("Unhandled CMIS type '" + baseTypeId + "', cannot get child at path " + path);
        }
    }

    @Override
    public CollectionIterable<AmetysObject> getChildren() throws AmetysRepositoryException
    {
        Collection<AmetysObject> aoChildren = new ArrayList<>(); 
        
        if (_session == null)
        {
            return new CollectionIterable<>(aoChildren);
        }
        
        ItemIterable<CmisObject> children = _root.getChildren();
        
        for (CmisObject child : children)
        {
            BaseTypeId typeId = child.getBaseTypeId();
            
            if (typeId.equals(BaseTypeId.CMIS_FOLDER))
            {
                aoChildren.add(new CMISResourcesCollection((Folder) child, this, this));
            }
            else if (typeId.equals(BaseTypeId.CMIS_DOCUMENT))
            {
                Document cmisDoc = (Document) child;
                
                // Check if CMIS document has content, if not ignore it
                if (StringUtils.isNotEmpty(cmisDoc.getContentStreamFileName()))
                {
                    aoChildren.add(new CMISResource(cmisDoc, this, this));
                }
            }
            else
            {
                __LOGGER.warn("Unhandled CMIS type {}. It will be ignored.", typeId);
            }
        }

        return new CollectionIterable<>(aoChildren);
    }

    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        if (_session == null)
        {
            return false;
        }
        
        ItemIterable<CmisObject> children = _root.getChildren();
        for (CmisObject child : children)
        {
            if (child.getName().equals(name))
            {
                return true;
            }
        }
        
        return false;
    }
    
    public ModifiableModelLessDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_factory.getDataTypesExtensionPoint(), repositoryData);
    }
    
    @Override
    public ModifiableCompositeMetadata getMetadataHolder()
    {
        return new JCRCompositeMetadata(getNode(), _factory._resolver);
    }
    
    @Override
    public String getIconCls()
    {
        if (_session == null)
        {
            return "ametysicon-share40 decorator-ametysicon-sign-caution a-tree-decorator-error-color";
        }
        
        String productName = _session.getRepositoryInfo().getProductName();
        if (productName.toLowerCase().indexOf("alfresco") != -1)
        {
            // FIXME EXPLORER-494 Use a dedicated Alfresco glyph
            return "ametysicon-share40";
        }
        else if (productName.toLowerCase().indexOf("nuxeo") != -1)
        {
            // FIXME EXPLORER-494 Use a dedicated Nuxeo glyph
            return "ametysicon-share40";
        }
        
        // FIXME EXPLORER-494 Use a dedicated CMIS glyph
        return "ametysicon-share40";
    }

    @Override
    public String getApplicationId()
    {
        return APPLICATION_ID;
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        return _name;
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        if (_parentPath == null)
        {
            _parentPath = getParent().getPath();
        }
        
        return _parentPath;
    }

    @Override
    public String getPath() throws AmetysRepositoryException
    {
        return getParentPath() + "/" + getName();
    }
    
    @Override
    public Node getNode()
    {
        return _node;
    }
    
    @Override
    public String getId()
    {
        try
        {
            return _factory.getScheme() + "://" + _node.getIdentifier();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get node UUID", e);
        }
    }
    
    public boolean hasChildResources() throws AmetysRepositoryException
    {
        // we don't actually know if there are children or not, 
        // but it's an optimization to don't make another CMIS request
        return true;
    }
    
    public boolean hasChildExplorerNodes() throws AmetysRepositoryException
    {
        // we don't actually know if there are children or not, 
        // but it's an optimization to don't make another CMIS request
        return true;
    }
    
    @Override
    public void rename(String newName) throws AmetysRepositoryException
    {
        try
        {
            getNode().getSession().move(getNode().getPath(), getNode().getParent().getPath() + "/" + newName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    @Override
    public void remove() throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        try
        {
            getNode().remove();
        }
        catch (ConstraintViolationException e)
        {
            throw new RepositoryIntegrityViolationException(e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public <A extends AmetysObject> A getParent() throws AmetysRepositoryException
    {
        return (A) _factory.getParent(this);
    }

    @Override
    public void saveChanges() throws AmetysRepositoryException
    {
        try
        {
            getNode().getSession().save();
        }
        catch (javax.jcr.RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to save changes", e);
        }
    }
    
    @Override
    public void revertChanges() throws AmetysRepositoryException
    {
        try
        {
            getNode().refresh(false);
        }
        catch (javax.jcr.RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to revert changes.", e);
        }
    }
    
    @Override
    public boolean needsSave() throws AmetysRepositoryException
    {
        try
        {
            return _node.getSession().hasPendingChanges();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    @Override
    public String getResourcePath() throws AmetysRepositoryException
    {
        return getExplorerPath();
    }
    
    @Override
    public String getExplorerPath()
    {
        AmetysObject parent = getParent();
        
        if (parent instanceof ExplorerNode)
        {
            return ((ExplorerNode) parent).getExplorerPath() + "/" + getName();
        }
        else
        {
            return "";
        }
    }
    
    /**
     * Get the user to connect to CMIS repository
     * @return the user login
     * @throws AmetysRepositoryException if an error occurred
     */
    public String getUser() throws AmetysRepositoryException
    {
        return getValue(DATA_USER);
    }
    
    /**
     * Get the password to connect to CMIS repository
     * @return the user password
     * @throws AmetysRepositoryException if an error occurred
     */
    public String getPassword() throws AmetysRepositoryException
    {
        return getValue(DATA_PASSWORD);
    }
    
    /**
     * Get the CMIS repository URL
     * @return the CMIS repository URL
     * @throws AmetysRepositoryException if an error occurred
     */
    public String getRepositoryUrl() throws AmetysRepositoryException
    {
        return getValue(DATA_REPOSITORY_URL);
    }
    
    /**
     * Get the CMIS repository id
     * @return the CMIS repository id
     * @throws AmetysRepositoryException if an error occurred
     */
    public String getRepositoryId () throws AmetysRepositoryException
    {
        return getValue(DATA_REPOSITORY_ID);
    }
    
    /**
     * Get the CMIS mount point
     * @return the mount point
     * @throws AmetysRepositoryException if an error occurred
     */
    public String getMountPoint() throws AmetysRepositoryException
    {
        return getValue(DATA_MOUNT_POINT);
    }
    
    /**
     * Set the URL of the CMIS repository
     * @param url the CMIS repository URL
     * @throws AmetysRepositoryException if an error occurred
     */
    public void setRepositoryUrl(String url) throws AmetysRepositoryException
    {
        setValue(DATA_REPOSITORY_URL, url);
    }
    
    /**
     * Set the id of the CMIS repository
     * @param id the CMIS repository id
     * @throws AmetysRepositoryException if an error occurred
     */
    public void setRepositoryId(String id) throws AmetysRepositoryException
    {
        setValue(DATA_REPOSITORY_ID, id);
    }
    
    /**
     * Set a mount point for the CMIS Repository
     * @param mountPoint the mount point path
     */
    public void setMountPoint(String mountPoint) 
    {
        setValue(DATA_MOUNT_POINT, mountPoint);
    }
    
    /**
     * Set a user name for the CMIS Repository
     * @param user the login
     */
    public void setUser(String user) 
    {
        setValue(DATA_USER, user);
    }
    
    /**
     * Set a password for the CMIS Repository
     * @param password the password
     */
    public void setPassword(String password) 
    {
        setValue(DATA_PASSWORD, password);
    }
    
    @Override
    public String getDescription()
    {
        return null;
    }
}
