/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.cmis;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.chemistry.opencmis.client.api.CmisObject;
import org.apache.chemistry.opencmis.client.api.Document;
import org.apache.chemistry.opencmis.client.api.Folder;
import org.apache.chemistry.opencmis.client.api.ObjectId;
import org.apache.chemistry.opencmis.client.api.Session;
import org.apache.chemistry.opencmis.client.api.SessionFactory;
import org.apache.chemistry.opencmis.client.runtime.SessionFactoryImpl;
import org.apache.chemistry.opencmis.commons.SessionParameter;
import org.apache.chemistry.opencmis.commons.enums.BaseTypeId;
import org.apache.chemistry.opencmis.commons.enums.BindingType;
import org.apache.chemistry.opencmis.commons.exceptions.CmisBaseException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisConnectionException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisObjectNotFoundException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.jcr.JCRAmetysObjectFactory;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Create the Root of CMIS Resources Collections
 */
public class CMISTreeFactory extends AbstractLogEnabled implements JCRAmetysObjectFactory<AmetysObject>, Configurable, Serviceable, Initializable, Observer
{
    /** Nodetype for resources collection */
    public static final String CMIS_ROOT_COLLECTION_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":cmis-root-collection";
    
    private static final String __SESSION_CACHE = CMISTreeFactory.class.getName() + "$cmisSessionCache";
    
    /** The application {@link AmetysObjectResolver} */
    protected AmetysObjectResolver _resolver;
    
    /** The configured scheme */
    protected String _scheme;

    /** The configured nodetype */
    protected String _nodetype;
    
    /** JCR Repository */
    protected Repository _repository;
    
    private ObservationManager _observationManager;

    private AbstractCacheManager _cacheManager;
    
    private ModelItemTypeExtensionPoint _modelLessBasicTypesExtensionPoint;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _modelLessBasicTypesExtensionPoint = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_MODEL_LESS_BASIC);
    }

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _scheme = configuration.getChild("scheme").getValue();
        
        Configuration[] nodetypesConf = configuration.getChildren("nodetype");
        
        if (nodetypesConf.length != 1)
        {
            throw new ConfigurationException("A SimpleAmetysObjectFactory must have one and only one associated nodetype. "
                                           + "The '" + configuration.getAttribute("id") + "' component has " + nodetypesConf.length);
        }
        
        _nodetype = nodetypesConf[0].getValue();
    }
    
    public void initialize() throws Exception
    {
        _observationManager.registerObserver(this);
        _cacheManager.createMemoryCache(__SESSION_CACHE,
                new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_CACHE_CMIS_SESSION_LABEL"),
                new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_CACHE_CMIS_SESSION_DESCRIPTION"),
                true,
                null);
    }

    @Override
    public CMISRootResourcesCollection getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException, RepositoryException
    {
        CMISRootResourcesCollection root = new CMISRootResourcesCollection(node, parentPath, this);
        
        if (!root.hasValue(CMISRootResourcesCollection.DATA_REPOSITORY_URL))
        {
            // Object just created, can't connect right now
            return root;
        }
        
        try
        {
            Session session = getAtomPubSession(root);
            
            Folder rootFolder = null;
            if (session != null)
            {
                String mountPoint = root.getMountPoint();
                // mount point is the root folder
                if (StringUtils.isBlank(mountPoint) || Strings.CS.equals(mountPoint, "/"))
                {
                    rootFolder = session.getRootFolder();
                }
                // any other valid mount point
                else if (StringUtils.isNotBlank(mountPoint) && Strings.CS.startsWith(mountPoint, "/"))
                {
                    try
                    {
                        rootFolder = (Folder) session.getObjectByPath(mountPoint);
                    }
                    catch (CmisObjectNotFoundException e)
                    {
                        getLogger().error("The mount point '{}' can't be found in the remote repository {}", mountPoint, root.getRepositoryId(), e);
                    }
                }
                
                // the mount point is valid
                if (rootFolder != null)
                {
                    root.connect(session, rootFolder);
                }
            }
        }
        catch (CmisConnectionException e)
        {
            getLogger().error("Connection to CMIS Atom Pub service failed", e);
        }
        catch (CmisObjectNotFoundException e)
        {
            getLogger().error("The CMIS Atom Pub service url refers to a non-existent repository", e);
        }
        catch (CmisBaseException e)
        {
            // all others CMIS errors
            getLogger().error("An error occured during call of CMIS Atom Pub service", e);
        }
        
        return root;
    }

    @Override
    public AmetysObject getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // l'id est de la forme <scheme>://uuid(/<cmis_id)
        String uuid = id.substring(getScheme().length() + 3);
        int index = uuid.indexOf("/");
        
        if (index != -1)
        {
            CMISRootResourcesCollection root = getCMISRootResourceCollection (getScheme() + "://" + uuid.substring(0, index));
            Session session = root.getSession();
            if (session == null)
            {
                throw new UnknownAmetysObjectException("Connection to CMIS server failed");
            }
            
            ObjectId cmisID = session.createObjectId(uuid.substring(index + 1));
            CmisObject cmisObject = session.getObject(cmisID);
            
            BaseTypeId baseTypeId = cmisObject.getBaseTypeId();

            if (baseTypeId.equals(BaseTypeId.CMIS_FOLDER))
            {
                return new CMISResourcesCollection((Folder) cmisObject, root, null);
            }
            else if (baseTypeId.equals(BaseTypeId.CMIS_DOCUMENT))
            {
                Document cmisDoc = (Document) cmisObject;
                try
                {
                    // EXPLORER-243 alfresco's id point to a version, not to the real "live" document
                    if (!cmisDoc.isLatestVersion())
                    {
                        cmisDoc = cmisDoc.getObjectOfLatestVersion(false);
                    }
                }
                catch (CmisBaseException e)
                {
                    // EXPLORER-269 does nothing, nuxeo sometimes throws a CmisRuntimeException here
                }
                
                return new CMISResource(cmisDoc, root, null);
            }
            else
            {
                throw new IllegalArgumentException("Unhandled CMIS type: " + baseTypeId);
            }
        }
        else
        {
            return getCMISRootResourceCollection (id);
        }
    }
    
    @Override
    public AmetysObject getAmetysObjectById(String id, javax.jcr.Session session) throws AmetysRepositoryException, RepositoryException
    {
        return getAmetysObjectById(id);
    }
    
    /**
     * Retrieves an {@link CMISRootResourcesCollection}, given its id.<br>
     * @param id the identifier.
     * @return the corresponding {@link CMISRootResourcesCollection}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    protected CMISRootResourcesCollection getCMISRootResourceCollection (String id) throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode(id);
            
            if (!node.getPath().startsWith('/' + AmetysObjectResolver.ROOT_REPO))
            {
                throw new AmetysRepositoryException("Cannot resolve a Node outside Ametys tree");
            }
            
            return getAmetysObject(node, null);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get AmetysObject for id: " + id, e);
        }
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // l'id est de la forme <scheme>://uuid(/<cmis_id)
        String uuid = id.substring(getScheme().length() + 3);
        int index = uuid.indexOf("/");
        
        if (index != -1)
        {
            CMISRootResourcesCollection root = getCMISRootResourceCollection (uuid.substring(0, index));
            Session session = root.getSession();
            if (session == null)
            {
                return false;
            }
            
            ObjectId cmisID = session.createObjectId(uuid.substring(index + 1));
            
            return session.getObject(cmisID) == null;
        }
        else
        {
            try
            {
                getNode(id);
                return true;
            }
            catch (UnknownAmetysObjectException e)
            {
                return false;
            }
        }
    }

    public String getScheme()
    {
        return _scheme;
    }

    public Collection<String> getNodetypes()
    {
        return Collections.singletonList(_nodetype);
    }

    /**
     * Returns the parent of the given {@link AmetysObject} .
     * @param object a {@link AmetysObject}.
     * @return the parent of the given {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public AmetysObject getParent(CMISRootResourcesCollection object) throws AmetysRepositoryException
    {
        try
        {
            Node node = object.getNode();
            Node parentNode = node.getParent();
        
            return _resolver.resolve(parentNode, false);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to retrieve parent object of object " + object.getName(), e);
        }
    }

    /**
     * Returns the JCR Node associated with the given object id.<br>
     * This implementation assumes that the id is like <code>&lt;scheme&gt;://&lt;uuid&gt;</code>
     * @param id the unique id of the object
     * @return the JCR Node associated with the given id
     */
    protected Node getNode(String id)
    {
        // id = <scheme>://<uuid>
        String uuid = id.substring(getScheme().length() + 3);
        
        javax.jcr.Session session = null;
        try
        {
            session = _repository.login();
            Node node = session.getNodeByIdentifier(uuid);
            return node;
        }
        catch (ItemNotFoundException e)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new UnknownAmetysObjectException("There's no node for id " + id, e);
        }
        catch (RepositoryException e)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("Unable to get AmetysObject for id: " + id, e);
        }
    }
    
    /**
     * Opening a Atom Pub Connection
     * @param root the JCR root folder
     * @return The created session or <code>null</code> if connection to CMIS server failed
     */
    public Session getAtomPubSession(CMISRootResourcesCollection root)
    {
        Cache<String, Session> sessionCache = _cacheManager.get(__SESSION_CACHE);
        String rootId = root.getId();
        
        return sessionCache.get(rootId, key -> _getAtomPubSession(root));
    }

    private Session _getAtomPubSession(CMISRootResourcesCollection root)
    {
        String url = root.getRepositoryUrl();
        String user = root.getUser();
        String password = root.getPassword();
        String repositoryId = root.getRepositoryId();
        
        try
        {
            Map<String, String> params = new HashMap<>();

            // user credentials
            params.put(SessionParameter.USER, user);
            params.put(SessionParameter.PASSWORD, password);

            // connection settings
            params.put(SessionParameter.ATOMPUB_URL, url);
            params.put(SessionParameter.BINDING_TYPE, BindingType.ATOMPUB.value());
            
            if (StringUtils.isEmpty(repositoryId))
            {
                SessionFactory f = SessionFactoryImpl.newInstance();
                List<org.apache.chemistry.opencmis.client.api.Repository> repositories = f.getRepositories(params);
                repositoryId = repositories.listIterator().next().getId();
                
                // save repository id for next times
                root.setRepositoryId(repositoryId);
                root.saveChanges();
            }
            
            params.put(SessionParameter.REPOSITORY_ID, repositoryId);
            
            // create session
            SessionFactory f = SessionFactoryImpl.newInstance();
            Session session = f.createSession(params);
            return session;
        }
        catch (CmisConnectionException e)
        {
            getLogger().error("Connection to CMIS Atom Pub service ({}) failed", url, e);
        }
        catch (CmisObjectNotFoundException e)
        {
            getLogger().error("The CMIS Atom Pub service url ({}) refers to a non-existent repository ({})", url, repositoryId, e);
        }
        catch (CmisBaseException e)
        {
            // all others CMIS errors
            getLogger().error("An error occured during call of CMIS Atom Pub service ({})", url, e);
        }
        
        return null;
    }
    
    public int getPriority()
    {
        return Observer.MAX_PRIORITY;
    }
    
    public boolean supports(Event event)
    {
        String eventType = event.getId();
        return ObservationConstants.EVENT_COLLECTION_DELETED.equals(eventType) || ObservationConstants.EVENT_CMIS_COLLECTION_UPDATED.equals(eventType);
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Cache<String, Session> sessionCache = _cacheManager.get(__SESSION_CACHE);
        String rootId = (String) event.getArguments().get(ObservationConstants.ARGS_ID);
        if (sessionCache.hasKey(rootId))
        {
            sessionCache.invalidate(rootId);
        }
    }
    
    /**
     * Retrieves the extension point with available data types for {@link JCRResourcesCollection}
     * @return the extension point with available data types for {@link JCRResourcesCollection}
     */
    public ModelItemTypeExtensionPoint getDataTypesExtensionPoint()
    {
        return _modelLessBasicTypesExtensionPoint;
    }
}
