/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.dublincore;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.Constants;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * This class handle the enumerated values for DublinCore metadata
 *
 */
public class DublinCoreMetadataProvider extends AbstractLogEnabled implements Component, Contextualizable, ThreadSafe, Initializable
{
    /** Avalon Role */
    public static final String ROLE = DublinCoreMetadataProvider.class.getName();
    
    private Map<String, Map<String, I18nizableText>> _entries;
    
    private org.apache.cocoon.environment.Context _cocoonContext;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _entries = new HashMap<>();
    }
    
    /**
     * Get the enumerated metadata names
     * @return the enumerated metadata names
     */
    public Set<String> getEnumeratedMetadataNames()
    {
        Set<String> metadataNames = new HashSet<>();
        
        File dcFile = new File (_cocoonContext.getRealPath("/WEB-INF/param/dublincore"));
        if (dcFile.exists())
        {
            File[] xmlFiles = dcFile.listFiles(new FileFilter()
            {
                @Override
                public boolean accept(File file)
                {
                    return file.isFile() && file.getName().toLowerCase().endsWith(".xml");
                }
            });
            
            for (File xmlFile : xmlFiles)
            {
                String filename = xmlFile.getName().toLowerCase();
                metadataNames.add(filename.substring(0, filename.lastIndexOf(".")));
            }
        }
        
        return metadataNames;
    }
    
    /**
     * Determines if the DublinCore metadata is enumerated
     * @param metadataName the metadata name
     * @return true if the DublinCore metadata is enumerated
     */
    public boolean isEnumerated (String metadataName)
    {
        File file = new File (_cocoonContext.getRealPath("/WEB-INF/param/dublincore/" + metadataName + ".xml"));
        return file.exists();
    }
    
    /**
     * Provides the enumerated values
     * @param metadataName the metadata name
     * @return the enumerated values. Can be null.
     */
    public Map<String, I18nizableText> getEntries (String metadataName)
    {
        if (!isEnumerated(metadataName))
        {
            return null;
        }
        
        try
        {
            if (!_entries.containsKey(metadataName))
            {
                _parseFile(metadataName);
            }
            
            return _entries.get(metadataName);
        }
        catch (ConfigurationException e)
        {
            getLogger().warn("Unable to get enumerated values of DublinCore metadata '" + metadataName + "'", e);
            return null;
        }
    }
    
    /**
     * Retrieves a single label from a value.
     * @param metadataName the metadata nale
     * @param value the value.
     * @return the label or <code>null</code> if not found.
     */
    public I18nizableText getEntry (String metadataName, String value)
    {
        if (!isEnumerated(metadataName))
        {
            return null;
        }
        
        try
        {
            if (!_entries.containsKey(metadataName))
            {
                _parseFile(metadataName);
            }
            
            return _entries.get(metadataName).get(value);
        }
        catch (ConfigurationException e)
        {
            getLogger().warn("Unable to get enumerated values of DublinCore metadata '" + metadataName + "'", e);
            return null;
        }
    }
    
    /**
     * Parses the enumerator.
     * @param metadataName the metadata name
     * @throws ConfigurationException if the configuration is not valid.
     */
    protected void _parseFile(String metadataName) throws ConfigurationException
    {
        File file = new File (_cocoonContext.getRealPath("/WEB-INF/param/dublincore/" + metadataName + ".xml"));
        
        if (file.exists())
        {
            try (InputStream is = new FileInputStream(file))
            {
                Configuration configuration = new DefaultConfigurationBuilder().build(is);
                
                Map<String, I18nizableText> entries = _parseEnumerator(configuration);
                _entries.put(metadataName, entries);
            }
            catch (Exception e)
            {
                throw new ConfigurationException("Unable to read the configuration file " + file.getName(), e);
            }
        }
    }
    
    /**
     * Parses the enumerator.
     * @param configuration the configuration to use
     * @return the enumerator values
     * @throws ConfigurationException if the configuration is not valid.
     */
    protected Map<String, I18nizableText> _parseEnumerator(Configuration configuration) throws ConfigurationException
    {
        Map<String, I18nizableText> entries = new HashMap<>();
        
        for (Configuration entryConfig : configuration.getChildren("entry"))
        {
            String value = entryConfig.getChild("value").getValue();
            I18nizableText label = null;
            
            if (entryConfig.getChild("label", false) != null)
            {
                label = _parseI18nizableText(entryConfig, "label");
            }
            
            entries.put(value, label);
        }
        
        return entries;
    }
    
    /**
     * Parses an i18n text.
     * @param config the configuration to use.
     * @param name the child name.
     * @return the i18n text.
     * @throws ConfigurationException if the configuration is not valid.
     */
    protected I18nizableText _parseI18nizableText(Configuration config, String name) throws ConfigurationException
    {
        return I18nizableText.parseI18nizableText(config.getChild(name), "application");
    }
    
    
}
