/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.dublincore;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.ServiceableReader;
import org.apache.commons.io.IOUtils;
import org.apache.excalibur.source.Source;
import org.xml.sax.SAXException;

import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Reader for RDF file of a {@link Resource}
 */
public class RDFResourceReader extends ServiceableReader
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The source resolver */
    protected org.apache.excalibur.source.SourceResolver _srcResolver;
    
    /** The resource */
    protected Resource _object;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _resolver = (AmetysObjectResolver) sManager.lookup(AmetysObjectResolver.ROLE);
        _srcResolver = (org.apache.excalibur.source.SourceResolver) sManager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
    }
    
    @Override
    public void setup(SourceResolver sResolver, Map objModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        super.setup(sResolver, objModel, src, par);
        
        String id = par.getParameter("id", null);
        String path = par.getParameter("path", null);
        
        assert id != null || path != null;
        
        if (id != null)
        {
            _object = _resolver.resolveById(id);
        }
        else
        {
            _object = _resolver.resolveByPath(path);
        }
    }
    
    @Override
    public long getLastModified()
    {
        return _object.getLastModified().getTime();
    }
    
    @Override
    public String getMimeType()
    {
        return "application/rdf+xml";
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Source src = null;
        try
        {
            String name = _object.getName();
            name = name.substring(0, name.lastIndexOf(".")) + ".rdf";
            name = name.replaceAll("\\\\", "\\\\\\\\");
            name = name.replaceAll("\\\"", "\\\\\\\"");
            
            Response response = ObjectModelHelper.getResponse(objectModel);
            response.setHeader("Content-Disposition", "attachment; filename=\"" + name + "\"");
            
            src = _srcResolver.resolveURI("cocoon://plugins/explorer/dublincore/rdf?id=" + _object.getId(), null, new HashMap<>());
            
            try (InputStream is = src.getInputStream())
            {
                response.setHeader("Content-Length", Long.toString(src.getContentLength()));
    
                IOUtils.copy(is, out);
            }
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to download file of id " + _object.getId(), e);
        }
        finally
        {
            IOUtils.closeQuietly(out);
            if (src != null)
            {
                _srcResolver.release(src);
            }
        }
    }
    
    @Override
    public void recycle()
    {
        super.recycle();
        _object = null;
    }
}
