/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.explorer.resources;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.servlet.multipart.Part;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationMode;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationResult;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;

/**
 * Adding or updating a resource file.
 */
public class AddOrUpdateResource implements Serviceable, Contextualizable, Component
{
    /** Avalon Role */
    public static final String ROLE = AddOrUpdateResource.class.getName();
    
    /** resource operation helper */
    protected AddOrUpdateResourceHelper _addOrUpdateResourceHelper;
    
    /** The avalon context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _addOrUpdateResourceHelper = (AddOrUpdateResourceHelper) serviceManager.lookup(AddOrUpdateResourceHelper.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Add, add and rename or update a resource
     * @param parentId The id of the collection parent
     * @param rawMode The mode
     * @param unzip If there is an unzip to be done 
     * @param parts The parts to import
     * @return The results
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = "Plugin_Explorer_File_Add", paramIndex = 0, rightContext = "right.assignment.context.resource")
    public Map addOrUpdateResources(String parentId, String rawMode, Boolean unzip, List<Part> parts)
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force the default workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            Map<String, Object> result = new HashMap<>();
            List<Map<String, Object>> resultsForFileSuccess = new ArrayList<>();
            
            for (Part part : parts)
            {
                Map<String, Object> resultForPart = new HashMap<>();
                String rawModeForPart = rawMode;
                
                if (unzip && part.getFileName().endsWith(".zip"))
                {
                    rawModeForPart = "add-unzip";
                }
                
                ResourceOperationMode mode = ResourceOperationMode.createsFromRawMode(rawModeForPart);
                
                // perform the add/update operation
                _addOrUpdateResourceHelper.checkAddResourceRight(parentId);
                ResourceOperationResult operationResult = _addOrUpdateResourceHelper.performResourceOperation(part, parentId, mode);
                
                if (!operationResult.isSuccess())
                {
                    Map<String, String> errors = (Map<String, String>) result.computeIfAbsent("message", l -> new HashMap<>());
                    errors.put(part.getFileName(), operationResult.getErrorMessage());
                }
                else
                {
                    result.put("parentID", parentId);
                    result.put("success", true);
                    
                    if (operationResult.isUnzip())
                    {
                        resultForPart.put("unzip", true);
                        resultForPart.put("reload", true);
                    }
                    else
                    {
                        Resource resource = operationResult.getResource();
                        resultForPart.put("id", resource.getId());
                        resultForPart.put("name", resource.getName());
                    }
                    resultsForFileSuccess.add(resultForPart);
                }
            }
            
            result.put("resultsForFileSuccess", resultsForFileSuccess);
            
            request.setAttribute(JSonReader.OBJECT_TO_READ, result);
            return result;
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
}
