/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.actions;

import java.io.IOException;
import java.io.InputStream;
import java.text.Normalizer;
import java.text.Normalizer.Form;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.servlet.multipart.RejectedPart;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;
import org.apache.commons.io.input.CloseShieldInputStream;
import org.apache.commons.lang.IllegalClassException;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.google.common.collect.Ordering;

/**
 * Dedicated helper in order to add or update an explorer resource
 */
public final class AddOrUpdateResourceHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role name */
    public static final String ROLE = AddOrUpdateResourceHelper.class.getName();
    
    /** The resource DAO */
    protected ExplorerResourcesDAO _resourcesDAO;
    /** The ametys resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    /** Observer manager. */
    protected ObservationManager _observationManager;

    /** The right manager */
    protected RightManager _rightManager;
    
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resourcesDAO = (ExplorerResourcesDAO) serviceManager.lookup(ExplorerResourcesDAO.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    /**
     *  Possible add and update modes
     */
    public enum ResourceOperationMode
    {
        /** Add */
        ADD("add"),
        /** Add with an unzip */
        ADD_UNZIP("add-unzip"),
        /** Add and allow rename */
        ADD_RENAME("add-rename"),
        /** Update */
        UPDATE("update");
        
        private String _mode;
        
        private ResourceOperationMode(String mode)
        {
            _mode = mode;
        }  
           
        @Override
        public String toString()
        {
            return _mode;
        }
        
        /**
         * Converts an raw input mode to the corresponding ResourceOperationMode
         * @param mode The raw mode to convert
         * @return the corresponding ResourceOperationMode or null if unknown
         */
        public static ResourceOperationMode createsFromRawMode(String mode)
        {
            for (ResourceOperationMode entry : ResourceOperationMode.values())
            {
                if (entry.toString().equals(mode))
                {
                    return entry;
                }
            }
            return null;
        }
    }
    
    /**
     * Check right to add resources
     * @param folderId the folder id to add resources
     */
    public void checkAddResourceRight(String folderId)
    {
        AmetysObject folder = _resolver.resolveById(folderId);
        if (!(folder instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, folder.getClass());
        }
        
        checkAddResourceRight((ModifiableResourceCollection) folder);
    }
    
    /**
     * Check right to add resources
     * @param folder the folder to add resources
     */
    public void checkAddResourceRight(ModifiableResourceCollection folder)
    {
        if (_rightManager.hasRight(_currentUserProvider.getUser(), ExplorerResourcesDAO.RIGHTS_RESOURCE_ADD, folder) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to add file without convenient right [" + ExplorerResourcesDAO.RIGHTS_RESOURCE_ADD + "]");
        }
    }
    
    /**
     * Perform an add or update resource operation
     * @param part The part representing the file for this operation
     * @param parentId The identifier of the parent collection
     * @param mode The operation mode
     * @return the result of the operation
     */
    public ResourceOperationResult performResourceOperation(Part part, String parentId, ResourceOperationMode mode)
    {
        try
        {
            AmetysObject object = _resolver.resolveById(parentId);
            if (!(object instanceof ModifiableResourceCollection))
            {
                throw new IllegalClassException(ModifiableResourceCollection.class, object.getClass());
            }
            
            return performResourceOperation(part, (ModifiableResourceCollection) object, mode);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().error("Unable to add file : the collection of id '{}' does not exist anymore", parentId, e);
            return new ResourceOperationResult("unknown-collection");
        }
    }
    
    /**
     * Perform an add or update resource operation
     * @param part The part representing the resource for this operation
     * @param parent The parent collection
     * @param mode The operation mode
     * @return the result of the operation
     */
    public ResourceOperationResult performResourceOperation(Part part, ModifiableResourceCollection parent, ResourceOperationMode mode)
    {
        if (part instanceof RejectedPart rejectedPart && rejectedPart.getMaxContentLength() == 0)
        {
            return new ResourceOperationResult("infected");
        }
        else if (part == null || part instanceof RejectedPart)
        {
            return new ResourceOperationResult("rejected");
        }
        
        if (!_resourcesDAO.checkLock(parent))
        {
            getLogger().warn("User '{}' is trying to modify the collection '{}' but it is locked by another user", _currentUserProvider.getUser(), parent.getName());
            return new ResourceOperationResult("locked");
        }
        
        String fileName = part.getUploadName();
        
        try (InputStream is = part.getInputStream())
        {
            return performResourceOperation(is, fileName, parent, mode);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to add file to the collection of id '{}'", parent.getId(), e);
            return new ResourceOperationResult("error");
        }
    }
    
    /**
     * Perform an add or update resource operation
     * @param inputStream The data for this operation
     * @param fileName file name requested
     * @param parent The parent collection
     * @param mode The operation mode
     * @return the result of the operation
     */
    public ResourceOperationResult performResourceOperation(InputStream inputStream, String fileName, ModifiableResourceCollection parent, ResourceOperationMode mode)
    {
        String usedFileName = fileName;
        
        if (!Normalizer.isNormalized(usedFileName, Form.NFC))
        {
            usedFileName = Normalizer.normalize(usedFileName, Form.NFC);
        }
        
        if (!_resourcesDAO.checkLock(parent))
        {
            getLogger().warn("User '{}' is trying to modify the collection '{}' but it is locked by another user", _currentUserProvider.getUser(), parent.getName());
            return new ResourceOperationResult("locked");
        }
        
        if (fileName.toLowerCase().endsWith(".zip") && ResourceOperationMode.ADD_UNZIP.equals(mode))
        {
            return _unzip(parent, inputStream);
        }
        
        ModifiableResource resource = null;
        
        // Rename existing
        if (parent.hasChild(usedFileName))
        {
            if (ResourceOperationMode.ADD_RENAME.equals(mode))
            {
                // Find a new name
                String[] f = usedFileName.split("\\.");
                int index = 1;
                while (parent.hasChild(usedFileName))
                {
                    usedFileName = f[0] + "-" + (index++) + '.' + f[1];
                }
                resource = _resourcesDAO.createResource(parent, usedFileName);
            }
            else if (ResourceOperationMode.UPDATE.equals(mode))
            {
                resource = parent.getChild(usedFileName);
            }
            else 
            {
                return new ResourceOperationResult("already-exist");
            }
        }
        // Add
        else
        {
            resource = _resourcesDAO.createResource(parent, usedFileName);
        }
        
        try
        {
            if (!_resourcesDAO.checkLock(resource))
            {
                getLogger().warn("User '{}' is trying to modify the resource '{}' but it is locked by another user", _currentUserProvider.getUser(), parent.getName());
                return new ResourceOperationResult("locked-file");
            }
            
            _resourcesDAO.updateResource(resource, inputStream, fileName);
            parent.saveChanges();
            _resourcesDAO.checkpoint(resource);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to add file to the collection of id '{}'", parent.getId(), e);
            return new ResourceOperationResult("error");
        }
        
        // Notify listeners
        if (ResourceOperationMode.UPDATE.equals(mode))
        {
            _notifyResourcesUpdated(parent, resource);
        }
        else
        {
            _notifyResourcesCreated(parent, Collections.singletonList(resource));
        }
        
        return new ResourceOperationResult(resource);
    }
    
    /**
     * Fire the {@link ObservationConstants#EVENT_RESOURCE_CREATED} event
     * @param parent The parent collection of the resource
     * @param resources The created resources
     */
    protected void _notifyResourcesCreated(ModifiableResourceCollection parent, List<Resource> resources)
    {
        Map<String, Object> eventParams = new HashMap<>();
        
        // ARGS_RESOURCES (transform to a map while keeping iteration order)
        Map<String, Resource> resourceMap = resources.stream()
                .collect(Collectors.toMap(
                    Resource::getId, // key = id
                    Function.identity(), // value = resource
                    (u, v) -> u, // allow duplicates
                    LinkedHashMap::new // to respect iteration order
                ));
        
        eventParams.put(ObservationConstants.ARGS_RESOURCES, resourceMap);
        
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_PATH, parent.getPath());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_CREATED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Fire the {@link ObservationConstants#EVENT_RESOURCE_UPDATED} event
     * @param parent The parent collection of the resource
     * @param resource The updated resource
     */
    protected void _notifyResourcesUpdated(ModifiableResourceCollection parent, Resource resource)
    {
        Map<String, Object> eventParams = new HashMap<>();
        
        eventParams.put(ObservationConstants.ARGS_ID, resource.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, resource.getName());
        eventParams.put(ObservationConstants.ARGS_PATH, resource.getPath());
        eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, resource.getResourcePath());
        
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Unzip an inputStream and add the content to the resource collection
     * @param collection The collection where to unzip
     * @param inputStream the inputStream of data we want to unzip
     * @return messages
     */
    private ResourceOperationResult _unzip(ModifiableResourceCollection collection, InputStream inputStream)
    {
        try (ZipArchiveInputStream zipInputStream = new ZipArchiveInputStream(inputStream, "cp437"))
        {
            List<Resource> extractedResources = _unzip(collection, zipInputStream);
            
            // Notify listeners
            _notifyResourcesCreated(collection, extractedResources);
            
            return new ResourceOperationResult(extractedResources);
        }
        catch (IOException e) 
        {
            getLogger().error("Unable to unzip file", e);
            return new ResourceOperationResult("unzip-error");
        }
    }
    
    private List<Resource> _unzip(ModifiableResourceCollection collection, ZipArchiveInputStream zipInputStream) throws IOException
    {
        List<Resource> extractedResources = new LinkedList<>();
        
        ArchiveEntry zipEntry;
        while ((zipEntry = zipInputStream.getNextEntry()) != null)
        {
            ModifiableResourceCollection parentCollection = collection;
            
            String zipName = zipEntry.getName();
            String[] path = zipName.split("/");
            
            for (int i = 0; i < path.length - 1; i++)
            {
                String name = path[i];
                parentCollection = _addCollection(parentCollection, name);
            }
            
            String name = path[path.length - 1];
            if (zipEntry.isDirectory())
            {
                parentCollection = _addCollection(parentCollection, name);
            }
            else
            {
                // because of the getNextEntry() call, zipInputStream is restricted to the data of the entry
                Resource resource = _addZipEntry(parentCollection, zipInputStream, name);
                extractedResources.add(resource);
            }

        }
        
        // sort by resource names
        Ordering<Resource> resourceNameOrdering = Ordering.natural().onResultOf(Resource::getName);
        extractedResources.sort(resourceNameOrdering);
        
        return extractedResources;
    }
    
    private ModifiableResourceCollection _addCollection (ModifiableResourceCollection collection, String name)
    {
        if (collection.hasChild(name))
        {
            return collection.getChild(name);
        }
        else
        {
            ModifiableResourceCollection child = collection.createChild(name, collection.getCollectionType());
            collection.saveChanges();
            return child;
        }
    }
    
    private Resource _addZipEntry (ModifiableResourceCollection collection, InputStream zipInputStream, String fileName)
    {
        ModifiableResource resource;
        
        if (collection.hasChild(fileName))
        {
            resource = collection.getChild(fileName);
        }
        else
        {
            resource = _resourcesDAO.createResource(collection, fileName);
        }
        // the call to updateResource will close the InputStream.
        // We don't want the InputStream to be closed because it's a zipInputStream
        // containing all the zip data, not just this entry.
        try (CloseShieldInputStream csis = CloseShieldInputStream.wrap(zipInputStream))
        {
            _resourcesDAO.updateResource(resource, csis, fileName);
        }
        
        collection.saveChanges();

        _resourcesDAO.checkpoint(resource);
        
        return resource;
    }
    
    /**
     * Class representing the result of a resource operation.
     */
    public static class ResourceOperationResult
    {
        /** The created or updated resource(s) */
        private final List<Resource> _resources;
        /** Indicates if an unzip operation was executed */
        private final boolean _unzip;
        /** Indicates if the operation was successful */
        private final boolean _success;
        /** Type of error in case of unsuccessful operation */
        private final String _errorMessage;
        
        /**
         * constructor in case of a successful operation
         * @param resource The resource of this operation
         */
        protected ResourceOperationResult(Resource resource)
        {
            _resources = Collections.singletonList(resource);
            _unzip = false;
            
            _success = true;
            _errorMessage = null;
        }
        
        /**
         * constructor in case of a successful unzip operation
         * @param resources The list of resource for this operation
         */
        protected ResourceOperationResult(List<Resource> resources)
        {
            _resources = resources;
            _unzip = true;
            
            _success = true;
            _errorMessage = null;
        }
        
        /**
         * constructor in case of an error
         * @param errorMessage The error message.
         */
        protected ResourceOperationResult(String errorMessage)
        {
            _errorMessage = errorMessage;
            _success = false;
            
            _resources = null;
            _unzip = false;
        }
        
        /**
         * Retrieves the resource
         * Note that {@link #getResources()} should be used in case of an unzip. 
         * @return the resource
         */
        public Resource getResource()
        {
            return _resources.get(0);
        }
        
        /**
         * Retrieves the list of resources, in case of an unzip.
         * @return the resource
         */
        public List<Resource> getResources()
        {
            return _resources;
        }
        
        /**
         * Retrieves the unzip
         * @return the unzip
         */
        public boolean isUnzip()
        {
            return _unzip;
        }

        /**
         * Retrieves the success
         * @return the success
         */
        public boolean isSuccess()
        {
            return _success;
        }

        /**
         * Retrieves the errorMessage
         * @return the errorMessage
         */
        public String getErrorMessage()
        {
            return _errorMessage;
        }
    }
}
