/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.actions;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Context;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.IllegalClassException;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.util.Text;
import org.apache.tika.metadata.Metadata;

import org.ametys.core.file.TikaProvider;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.cmis.CMISRootResourcesCollection;
import org.ametys.plugins.explorer.cmis.CMISTreeFactory;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.generators.ResourcesExplorerGenerator;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollectionFactory;
import org.ametys.plugins.explorer.resources.metadata.populate.ResourceMetadataPopulator;
import org.ametys.plugins.explorer.resources.metadata.populate.ResourceMetadataPopulatorExtensionPoint;
import org.ametys.plugins.explorer.rights.ResourceRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.dublincore.DublinCoreAwareAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.lock.LockHelper;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Explorer resources DAO
 */
public class ExplorerResourcesDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = ExplorerResourcesDAO.class.getName();

    /** Right id to unlock all resources */
    public static final String RIGHTS_RESOURCE_UNLOCK_ALL = "Plugin_Explorer_File_Unlock_All";

    /** Right id to add a resource */
    public static final String RIGHTS_RESOURCE_ADD = "Plugin_Explorer_File_Add";

    /** Right id to rename a resource */
    public static final String RIGHTS_RESOURCE_RENAME = "Plugin_Explorer_File_Rename";

    /** Right id to delete a resource */
    public static final String RIGHTS_RESOURCE_DELETE = "Plugin_Explorer_File_Delete";

    /** Right id to edit DC metadata of a resource */
    public static final String RIGHTS_RESOURCE_EDIT_DC = "Plugin_Explorer_File_Edit_DC_Metadata";

    /** Right id to moderate comment a resource */
    public static final String RIGHTS_RESOURCE_MODERATE_COMMENT = "Plugin_Explorer_File_Moderate_Comments";

    /** Right id to add CMIS collection */
    public static final String RIGHTS_COLLECTION_CMIS_ADD = "Plugin_Explorer_CMIS_Add";

    /** Right id to add a folder */
    public static final String RIGHTS_COLLECTION_ADD = "Plugin_Explorer_Folder_Add";

    /** Right id to edit a folder */
    public static final String RIGHTS_COLLECTION_EDIT = "Plugin_Explorer_Folder_Edit";

    /** Right id to delete a folder */
    public static final String RIGHTS_COLLECTION_DELETE = "Plugin_Explorer_Folder_Delete";
    
    /** Ametys resolver */
    protected AmetysObjectResolver _resolver;

    /** The rights manager */
    protected RightManager _rightManager;

    /** Observer manager. */
    protected ObservationManager _observationManager;

    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;

    /** The avalon context */
    protected org.apache.avalon.framework.context.Context _context;

    /** The cocoon context */
    protected Context _cocoonContext;

    /** The tika provider. */
    protected TikaProvider _tikaProvider;

    /** The metadata populator extension point. */
    protected ResourceMetadataPopulatorExtensionPoint _metadataPopulatorEP;

    /** The users manager */
    protected UserHelper _userHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _tikaProvider = (TikaProvider) manager.lookup(TikaProvider.ROLE);
        _metadataPopulatorEP = (ResourceMetadataPopulatorExtensionPoint) manager.lookup(ResourceMetadataPopulatorExtensionPoint.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
    }

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    /**
     * Get the root nodes for resources
     * 
     * @return the root nodes
     */
    public List<ExplorerNode> getResourcesRootNodes()
    {
        List<ExplorerNode> roots = new ArrayList<>();
        roots.add(_resolver.resolveByPath(RepositoryConstants.NAMESPACE_PREFIX + ":resources"));
        return roots;
    }

    /**
     * Retrieves the set of standard data for the explorer root node
     * 
     * @return The data structured in a map
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, Object>> getRootNodesInfo()
    {
        // Assume that no read access is checked (required to display resource tree)
        List<Map<String, Object>> infos = new ArrayList<>();

        List<ExplorerNode> rootNodes = getResourcesRootNodes();
        for (ExplorerNode rootNode : rootNodes)
        {
            infos.add(getDefaultInfoAsRootNode(rootNode));
        }
        return infos;
    }

    /**
     * Get the necessary default info for a root node. Can be used to construct
     * a root node for a tree (client side).
     * 
     * @param id The root node id
     * @return A map which contains a set of default info (such as id,
     *         applicationId, path, type etc...)
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getDefaultInfoAsRootNode(String id)
    {
        // Assume that no read access is checked (required to display resource tree)
        ExplorerNode node = _resolver.resolveById(id);
        return getDefaultInfoAsRootNode(node);
    }

    /**
     * Get the necessary default info for a root node. Can be used to construct
     * a root node for a tree (client side).
     * 
     * @param rootNode The root node
     * @return A map which contains a set of default info (such as id,
     *         applicationId, path, type etc...)
     */
    public Map<String, Object> getDefaultInfoAsRootNode(ExplorerNode rootNode)
    {
        Map<String, Object> result = new HashMap<>();

        result.put("id", rootNode.getId());
        result.put("applicationId", rootNode.getApplicationId());
        result.put("name", "resources");
        result.put("cls", "root");
        result.put("iconCls", "ametysicon-folder249");

        result.put("text", getRootNodeLabel(rootNode));

        result.put("path", "/dummy/resources");
        result.put("type", ResourcesExplorerGenerator.RESOURCE_COLLECTION);

        boolean hasResources = false;
        if (rootNode instanceof ResourceCollection)
        {
            hasResources = ((ResourceCollection) rootNode).hasChildResources();
        }
        boolean hasChildNodes = rootNode.hasChildExplorerNodes();

        if (hasChildNodes)
        {
            result.put("hasChildNodes", true);
        }

        if (hasResources)
        {
            result.put("hasResources", true);
        }

        result.put("isModifiable", false);

        if (rootNode instanceof ModifiableExplorerNode)
        {
            result.put("canCreateChild", true);
        }

        return result;
    }

    /**
     * Get the root node label
     * @param rootNode the node
     * @return the readable label
     */
    public I18nizableText getRootNodeLabel(ExplorerNode rootNode)
    {
        return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_ROOT_NODE");
    }

    /**
     * Get the informations on given nodes (resources or collections)
     * 
     * @param ids The ids of node
     * @return the nodes information
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getNodesInfo(List<String> ids)
    {
        // Assume that no read access is checked (required for bus message target)
        List<Map<String, Object>> objects = new ArrayList<>();
        List<String> objectsNotFound = new ArrayList<>();

        for (String id : ids)
        {
            try
            {
                AmetysObject ao = _resolver.resolveById(id);

                if (ao instanceof ExplorerNode)
                {
                    objects.add(getExplorerNodeProperties((ExplorerNode) ao));
                }
                else if (ao instanceof Resource)
                {
                    objects.add(getResourceProperties((Resource) ao));
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                objectsNotFound.add(id);
            }
        }

        Map<String, Object> result = new HashMap<>();
        result.put("objects", objects);
        result.put("objectsNotFound", objectsNotFound);

        return result;
    }

    /**
     * Get the explorer node properties
     * 
     * @param node The explorer node
     * @return The properties
     */
    public Map<String, Object> getExplorerNodeProperties(ExplorerNode node)
    {
        Map<String, Object> infos = new HashMap<>();

        ExplorerNode root = node;
        AmetysObject parent = null;

        while (true)
        {
            parent = root.getParent();
            if (parent instanceof ExplorerNode)
            {
                root = (ExplorerNode) parent;
            }
            else
            {
                break;
            }
        }

        parent = node.getParent();

        infos.put("rootId", root.getId());
        infos.put("rootOwnerType", "explorer");
        infos.put("parentId", parent instanceof ExplorerNode ? parent.getId() : null);
        infos.put("id", node.getId());
        infos.put("applicationId", node.getApplicationId());
        infos.put("name", node.getName());
        infos.put("path", node.getExplorerPath());
        infos.put("isModifiable", node instanceof ModifiableAmetysObject);
        infos.put("canCreateChild", node instanceof ModifiableExplorerNode);
        infos.put("cls", node.getIconCls());

        infos.put("rights", getUserRights(node));

        return infos;
    }

    /**
     * Get the resource properties
     * 
     * @param resource The resources
     * @return The properties
     */
    public Map<String, Object> getResourceProperties(Resource resource)
    {
        Map<String, Object> infos = new HashMap<>();

        ResourceCollection parentAO = resource.getParent();
        ResourceCollection root = parentAO;
        while (true)
        {
            if (root.getParent() instanceof ResourceCollection)
            {
                root = root.getParent();
            }
            else
            {
                break;
            }
        }
        infos.put("id", resource.getId());
        infos.put("rootId", root.getId());
        infos.put("rootOwnerType", "explorer");
        infos.put("parentId", parentAO.getId());
        infos.put("name", resource.getName());
        infos.put("path", resource.getResourcePath());
        infos.put("isModifiable", resource instanceof ModifiableAmetysObject);

        infos.put("rights", getUserRights(parentAO));

        return infos;
    }
    
    /**
     * Get the pattern that match an explorer node path
     * @return the patterns
     */
    public Set<Pattern> getExplorerNodePathPatterns()
    {
        return Collections.singleton(Pattern.compile("^/" + RepositoryConstants.NAMESPACE_PREFIX + ":resources(/.*)?$"));
    }
    
    /**
     * Get the path of pages which match filter regexp
     * @param id The id of explorer node to start search
     * @param value the value to match
     * @param allowedExtensions The allowed file extensions (lower-case). Can be null or empty to not filter on file extensions
     * @return the matching paths
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<String> filterResourcesByRegExp(String id, String value, List<String> allowedExtensions)
    {
        // Assume that no read access is checked (required to filter resource tree)
        List<String> matchingPaths = new ArrayList<>();

        ExplorerNode root = _resolver.resolveById(id);
        
        String toMatch = org.apache.commons.lang3.StringUtils.stripAccents(value.toLowerCase()).trim();
        
        if (root instanceof TraversableAmetysObject)
        {
            TraversableAmetysObject traversableObject = (TraversableAmetysObject) root;
            AmetysObjectIterable< ? extends AmetysObject> children = traversableObject.getChildren();
            for (AmetysObject ao : children)
            {
                if (ao instanceof Resource)
                {
                    _getMatchingResource((Resource) ao, toMatch, allowedExtensions, matchingPaths);
                }
                else if (ao instanceof ExplorerNode)
                {
                    _getMatchingExplorerNode ((ExplorerNode) ao, toMatch, allowedExtensions, matchingPaths);
                }
            }
        }
        
        return matchingPaths;
    }
    
    private void _getMatchingExplorerNode(ExplorerNode explorerNode, String value, List<String> allowedExtensions, List<String> matchingPaths)
    {
        String title =  org.apache.commons.lang3.StringUtils.stripAccents(explorerNode.getName().toLowerCase());
        
        if (title.contains(value))
        {
            matchingPaths.add(explorerNode.getExplorerPath());
        }
        
        if (explorerNode instanceof TraversableAmetysObject)
        {
            TraversableAmetysObject traversableObject = (TraversableAmetysObject) explorerNode;
            
            AmetysObjectIterable< ? extends AmetysObject> children = traversableObject.getChildren();
            for (AmetysObject ao : children)
            {
                if (ao instanceof Resource)
                {
                    _getMatchingResource((Resource) ao, value, allowedExtensions, matchingPaths);
                }
                else if (ao instanceof ExplorerNode)
                {
                    _getMatchingExplorerNode ((ExplorerNode) ao, value, allowedExtensions, matchingPaths);
                }
            }
        }
    }
    
    private void _getMatchingResource(Resource resource, String value, List<String> allowedExtensions, List<String> matchingPaths)
    {
        String filename =  org.apache.commons.lang3.StringUtils.stripAccents(resource.getName().toLowerCase());
        String fileExtension = filename.lastIndexOf(".") > 0 ? filename.substring(filename.lastIndexOf(".") + 1) : "";
        if (filename.contains(value) && (allowedExtensions == null || allowedExtensions.size() == 0 || allowedExtensions.contains(fileExtension)))
        {
            matchingPaths.add(resource.getResourcePath());
        }
    }
    
    /**
     * Get the user rights on the resource collection
     * 
     * @param node The explorer node
     * @return The user's rights
     */
    protected Set<String> getUserRights(ExplorerNode node)
    {
        return _rightManager.getUserRights(_currentUserProvider.getUser(), node);
    }

    /**
     * Check current user right on given explorer node
     * 
     * @param id The id of the explorer node
     * @param rightId The if of right to check
     * @return true if user has right
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public boolean hasRight(String id, String rightId)
    {
        UserIdentity user = _currentUserProvider.getUser();
        ExplorerNode node = _resolver.resolveById(id);

        return _rightManager.hasRight(user, rightId, node) == RightResult.RIGHT_ALLOW;
    }

    /**
     * Check lock on a Ametys object
     * 
     * @param ao the Ametys object
     * @return <code>false</code> if the Ametys object is locked and can not be
     *         edited or deleted
     */
    public boolean checkLock(AmetysObject ao)
    {
        if (ao instanceof LockableAmetysObject)
        {
            LockableAmetysObject lockableAO = (LockableAmetysObject) ao;
            if (lockableAO.isLocked())
            {
                if (!LockHelper.isLockOwner(lockableAO, _currentUserProvider.getUser()))
                {
                    return false;
                }

                if (ao instanceof JCRAmetysObject)
                {
                    _addLockToken(((JCRAmetysObject) ao).getNode());
                }
            }

        }

        return true;
    }

    private void _addLockToken(Node node)
    {
        try
        {
            if (node.isLocked())
            {
                LockManager lockManager = node.getSession().getWorkspace().getLockManager();

                Lock lock = lockManager.getLock(node.getPath());
                Node lockHolder = lock.getNode();

                lockManager.addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add lock token", e);
        }
    }

    /**
     * Check the user privilege on object
     * 
     * @param object the object
     * @param rightId the right id
     * @throws IllegalAccessException if the user has no sufficient rights
     */
    public void checkUserRight(AmetysObject object, String rightId) throws IllegalAccessException
    {
        ExplorerNode node;
        if (object instanceof Resource)
        {
            node = object.getParent();
        }
        else
        {
            node = (ExplorerNode) object;
        }

        if (_rightManager.hasRight(_currentUserProvider.getUser(), rightId, object) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to access a privilege feature without convenient right [" + rightId
                    + ", /resources" + node.getExplorerPath() + "]");
        }
    }

    /**
     * Retrieve the rights for the user
     * 
     * @param user The user
     * @param right The right
     * @param object The object
     * @return True if the user has the right
     */
    public boolean getUserRight(UserIdentity user, String right, AmetysObject object)
    {
        AmetysObject explorerNode = object;
        while (explorerNode != null && !(explorerNode instanceof ExplorerNode))
        {
            explorerNode = explorerNode.getParent();
        }
        if (explorerNode == null)
        {
            return false;
        }
        return _rightManager.hasRight(user, right, explorerNode) == RightResult.RIGHT_ALLOW;
    }

    /**
     * Add a resource collection
     * 
     * @param parentId The identifier of the parent in which the resource
     *            collection will be added
     * @param desiredName The desired name for the resource collection
     * @param renameIfExists If false, in case of existing name the resource
     *            collection will not be created, if true it will be created and
     *            renamed
     * @return The result map with id, parentId, name and message keys
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addResourceCollection(String parentId, String desiredName, Boolean renameIfExists)
    {
        Map<String, Object> result = new HashMap<>();

        assert parentId != null;

        AmetysObject object = _resolver.resolveById(parentId);
        if (!(object instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, object.getClass());
        }

        List<String> errors = new LinkedList<>();
        ResourceCollection rc = addResourceCollection((ModifiableResourceCollection) object, desiredName, renameIfExists, errors);

        if (!errors.isEmpty())
        {
            result.put("message", errors.get(0));
        }
        else
        {
            result.put("id", rc.getId());
            result.put("parentID", parentId);
            result.put("name", rc.getName());
        }

        return result;
    }

    /**
     * Add a resource collection
     * 
     * @param parent The parent collection in which the resource collection will
     *            be added
     * @param desiredName The desired name for the resource collection
     * @param renameIfExists If false, in case of existing name the resource
     *            collection will not be created, if true it will be created and
     *            renamed
     * @param errors An optional list of possible error messages in case the
     *            creation failed. Possible values are: locked, already-exist.
     * @return The created resource collection or null if creation failed
     */
    public ResourceCollection addResourceCollection(ModifiableResourceCollection parent, String desiredName, Boolean renameIfExists, List<String> errors)
    {
        String originalName = desiredName;

        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_ADD, parent) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to add folder without convenient right [" + RIGHTS_COLLECTION_ADD + "]");
        }
        
        if (!checkLock(parent))
        {
            getLogger().warn("User '{}' try to modify collection '{}' but it is locked by another user", _currentUserProvider.getUser(), parent.getName());
            if (errors != null)
            {
                errors.add("locked");
            }
            return null;
        }

        if (!renameIfExists && parent.hasChild(originalName))
        {
            getLogger().warn("The object '{}' can not be renamed in '{}' : a object of same name already exists.", parent.getName(), originalName);
            if (errors != null)
            {
                errors.add("already-exist");
            }
            return null;
        }

        int index = 2;
        String name = originalName;
        while (parent.hasChild(name))
        {
            name = originalName + " (" + index + ")";
            index++;
        }

        ResourceCollection child = parent.createChild(name, getResourceCollectionType());
        parent.saveChanges();

        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, child.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, child.getName());
        eventParams.put(ObservationConstants.ARGS_PATH, child.getPath());

        _observationManager.notify(new Event(ObservationConstants.EVENT_COLLECTION_CREATED, _currentUserProvider.getUser(), eventParams));

        return child;
    }

    /**
     * Get the type of child resource collection
     * 
     * @return the type of child resource collection
     */
    public String getResourceCollectionType()
    {
        return JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE;
    }

    /**
     * Rename a resource, or resource collection
     * 
     * @param id The id of the object to rename
     * @param name The desired name to set to the object.
     * @return The result map with id, name and message keys
     * @throws RepositoryException If there is a repository error
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> renameObject(String id, String name) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();

        JCRAmetysObject object = _resolver.resolveById(id);
        List<String> errors = new LinkedList<>();
        JCRAmetysObject newObject = renameObject(object, name, errors);

        if (!errors.isEmpty())
        {
            String error = errors.get(0);
            result.put("message", error);
        }
        else
        {
            result.put("id", newObject.getId());
            result.put("name", name);
        }

        return result;
    }

    /**
     * Rename a resource, or resource collection
     * 
     * @param object The object to rename
     * @param name The desired name to set to the object.
     * @param errors An optional list of possible error messages in case the
     *            operation failed. Possible values are: locked, already-exist.
     * @return The new object
     * @throws RepositoryException If there is a repository error
     */
    public JCRAmetysObject renameObject(JCRAmetysObject object, String name, List<String> errors) throws RepositoryException
    {
        assert object != null;
        assert name != null;

        String legalName = Text.escapeIllegalJcrChars(name);

        // Check node is not the root node
        if ("ametys-internal:resources".equals(object.getName()))
        {
            throw new IllegalStateException("The resources root node can not be renamed !");
        }

        String rightId = object instanceof Resource ? RIGHTS_RESOURCE_RENAME : RIGHTS_COLLECTION_EDIT;
        if (_rightManager.hasRight(_currentUserProvider.getUser(), rightId, object) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to rename folder or file without convenient right [" + rightId + "]");
        }

        Node node = object.getNode();
        String oldName = object.getName();
        String oldObjectPath = object.getPath();

        if (!checkLock(object))
        {
            getLogger().warn("User '{}' is trying to rename object '{}' but it is locked by another user", _currentUserProvider.getUser(), object.getName());
            if (errors != null)
            {
                errors.add("locked");
            }
            return null;
        }
        else if (node.getParent().hasNode(legalName))
        {
            getLogger().warn("The object '{}' cannot be renamed in '{}' : an object with the same name already exists.", object.getName(), name);
            if (errors != null)
            {
                errors.add("already-exist");
            }
            return null;
        }
        else
        {
            String oldResourcePath = object instanceof Resource ? ((Resource) object).getResourcePath() : ((ResourceCollection) object).getExplorerPath();

            node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + legalName);
            node.getSession().save();

            // Resolve the new ametys object
            JCRAmetysObject newObject = (JCRAmetysObject) _resolver.resolve(node, false);

            // Notify listeners
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_ID, newObject.getId());
            eventParams.put(ObservationConstants.ARGS_PARENT_ID, newObject.getParent().getId());
            eventParams.put(ObservationConstants.ARGS_NAME, newObject.getName());
            eventParams.put(ObservationConstants.ARGS_PATH, newObject.getPath());
            eventParams.put("object.old.name", oldName);
            eventParams.put("object.old.path", oldObjectPath);

            if (object instanceof Resource)
            {
                eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, ((Resource) object).getResourcePath());
                eventParams.put("resource.old.path", oldResourcePath);
            }
            else
            {
                eventParams.put(ObservationConstants.ARGS_EXPLORER_PATH, ((ResourceCollection) object).getExplorerPath());
                eventParams.put("explorer.old.path", oldResourcePath);
            }

            _observationManager.notify(new Event(object instanceof JCRResource ? ObservationConstants.EVENT_RESOURCE_RENAMED : ObservationConstants.EVENT_COLLECTION_RENAMED,
                    _currentUserProvider.getUser(), eventParams));

            return newObject;
        }
    }

    /**
     * Delete an object
     * 
     * @param ids The list of identifiers for the objects to delete
     * @return The result map with a message key in case of an error
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteObject(List<String> ids)
    {
        assert ids != null;

        Map<String, Object> result = new HashMap<>();

        for (String id : ids)
        {
            RemovableAmetysObject object = (RemovableAmetysObject) _resolver.resolveById(id);
            List<String> errors = new LinkedList<>();
            deleteObject(object, errors);

            if (!errors.isEmpty())
            {
                String error = errors.get(0);
                result.put("message", error);
                result.put("success", false);
                return result;
            }
        }

        result.put("success", true);
        return result;
    }

    /**
     * Delete an object
     * 
     * @param object The object to delete
     * @param errors An optional list of possible error messages in case the
     *            creation failed. Possible values are: locked.
     * @return the parent id of the removed object
     */
    public String deleteObject(RemovableAmetysObject object, List<String> errors)
    {
        // Check node is not the root node
        if ("ametys-internal:resources".equals(object.getName()))
        {
            throw new IllegalStateException("The resources root node can not be deleted !");
        }
        
        String rightId = object instanceof Resource ? RIGHTS_RESOURCE_DELETE : RIGHTS_COLLECTION_DELETE;
        if (_rightManager.hasRight(_currentUserProvider.getUser(), rightId, object) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to delete file or folder without convenient right [" + rightId + "]");
        }

        if (!checkLock(object))
        {
            getLogger().warn("User '{}' is trying to delete object '{}' but it is locked by another user", _currentUserProvider.getUser(), object.getName());
            if (errors != null)
            {
                errors.add("locked");
            }
            return null;
        }

        ModifiableResourceCollection parent = object.getParent();
        String eventType = object instanceof Resource ? ObservationConstants.EVENT_RESOURCE_DELETED : ObservationConstants.EVENT_COLLECTION_DELETED;
        String parentId = parent.getId();

        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parentId);
        eventParams.put(ObservationConstants.ARGS_ID, object.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, object.getName());
        eventParams.put(ObservationConstants.ARGS_PATH, object.getPath());

        if (object instanceof Resource)
        {
            eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, ((Resource) object).getResourcePath());
        }
        else
        {
            eventParams.put(ObservationConstants.ARGS_EXPLORER_PATH, ((ResourceCollection) object).getExplorerPath());
            _observationManager.notify(new Event(ObservationConstants.EVENT_COLLECTION_DELETING, _currentUserProvider.getUser(), eventParams));
        }
        
        doDeleteObject(object);

        _observationManager.notify(new Event(eventType, _currentUserProvider.getUser(), eventParams));

        return parentId;
    }
    
    /**
     * Technically delete the object.
     * @param object the object to delete
     */
    protected void doDeleteObject(RemovableAmetysObject object)
    {
        ModifiableResourceCollection parent = object.getParent();
        object.remove();
        parent.saveChanges();
    }

    /**
     * Rename a resource
     * 
     * @param id The id of the resource to rename
     * @param name The desired name to set to the resource.
     * @return The result map with id, name and message keys
     * @throws RepositoryException If there is a repository error
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> renameResource(String id, String name) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();

        JCRResource resource = _resolver.resolveById(id);
        List<String> errors = new LinkedList<>();

        Resource newResource = renameResource(resource, name, errors);

        if (!errors.isEmpty())
        {
            String error = errors.get(0);
            result.put("message", error);
        }
        else
        {
            result.put("id", newResource.getId());
            result.put("name", name);
        }

        return result;
    }

    /**
     * Rename a resource
     * 
     * @param resource The resource to rename
     * @param name The desired name to set to the resource.
     * @param errors An optional list of possible error messages in case the
     *            operation failed. Possible values are: locked, already-exist.
     * @return The new resource
     * @throws RepositoryException If there is a repository error
     */
    public JCRResource renameResource(JCRResource resource, String name, List<String> errors) throws RepositoryException
    {
        assert resource != null;
        assert name != null;

        String legalName = Text.escapeIllegalJcrChars(name);

        // FIXME API getNode should use the new rename method
        String oldName = resource.getName();
        String oldPath = resource.getPath();
        String oldResourcePath = resource.getResourcePath();

        Node node = resource.getNode();

        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_RENAME, resource) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to rename file without convenient right [" + RIGHTS_RESOURCE_RENAME + "]");
        }

        // Check lock on resource
        if (!checkLock(resource))
        {
            getLogger().warn("User '{}' is trying to rename resource '{}' but it is locked by another user", _currentUserProvider.getUser(), resource.getName());
            if (errors != null)
            {
                errors.add("locked");
            }
            return null;
        }

        if (node.getParent().hasNode(legalName))
        {
            getLogger().warn("The resource '{}' cannot be renamed in '{}' : an object with the same name already exists.", resource.getName(), name);
            if (errors != null)
            {
                errors.add("already-exist");
            }
            return null;
        }

        String mimeType = _cocoonContext.getMimeType(legalName.toLowerCase());
        mimeType = mimeType == null ? "application/unknown" : mimeType;
        resource.setMimeType(mimeType);

        node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + legalName);

        node.getSession().save();

        // Resolve the new ametys object
        JCRResource newObject = (JCRResource) _resolver.resolve(node, false);

        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();

        eventParams.put(ObservationConstants.ARGS_ID, newObject.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, newObject.getParent().getId());
        eventParams.put(ObservationConstants.ARGS_NAME, newObject.getName());
        eventParams.put(ObservationConstants.ARGS_PATH, newObject.getPath());
        eventParams.put("object.old.name", oldName);
        eventParams.put("object.old.path", oldPath);

        eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, newObject.getResourcePath());
        eventParams.put("resource.old.path", oldResourcePath);

        _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_RENAMED, _currentUserProvider.getUser(), eventParams));

        return newObject;
    }

    /**
     * Copy file resources
     * 
     * @param ids The list of identifiers for the resources to copy
     * @param target The id of target to copy into
     * @return The result map with a message key in case of an error or with the
     *         list of uncopied/copied resources
     * @throws RepositoryException If there is a repository error
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> copyResource(List<String> ids, String target) throws RepositoryException
    {
        assert ids != null;
        assert target != null;

        AmetysObject object = _resolver.resolveById(target);
        if (!(object instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, object.getClass());
        }

        return copyResource(ids, (ModifiableResourceCollection) object);
    }

    /**
     * Copy file resources
     * 
     * @param ids The list of identifiers for the resources to copy
     * @param target The target to copy into
     * @return The result map with a message key in case of an error or with the
     *         list of uncopied/copied resources
     * @throws RepositoryException If there is a repository error
     */
    public Map<String, Object> copyResource(List<String> ids, ModifiableResourceCollection target) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        List<String> uncopiedResources = new ArrayList<>();
        List<String> copiedResourceIds = new ArrayList<>();

        assert ids != null;
        assert target != null;

        if (!checkLock(target))
        {
            getLogger().warn("User '{}' try to copy objet to '{}' but it is locked by another user", _currentUserProvider.getUser(), target.getName());
            result.put("message", "locked");
            return result;
        }

        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_ADD, target) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to copy file without convenient right [" + RIGHTS_RESOURCE_ADD + "]");
        }

        for (String id : ids)
        {
            Resource resourceToCopy = _resolver.resolveById(id);
            String fileName = resourceToCopy.getName();

            if (target.hasChild(fileName))
            {
                getLogger().warn("The resource '" + fileName + "' can not be copied : an object of same name already exists in the target collection");
                result.put("message", "already-exist");
                uncopiedResources.add(fileName);
            }
            else
            {
                ModifiableResource resource = createResource(target, fileName);

                try (InputStream is = resourceToCopy.getInputStream())
                {
                    updateResource(resource, is, fileName);
                }
                catch (IOException e)
                {
                    getLogger().warn("An error occurred while closing the ressource " + resource.getId(), e);
                }

                copiedResourceIds.add(resource.getId());
            }
        }

        target.saveChanges();

        for (String id : copiedResourceIds)
        {
            ModifiableResource resource = _resolver.resolveById(id);

            // Create version
            checkpoint(resource);

            // Notify listeners
            Map<String, Object> eventParams = new HashMap<>();
            Map<String, Resource> addedResource = new HashMap<>();
            addedResource.put(resource.getId(), resource);
            eventParams.put(ObservationConstants.ARGS_RESOURCES, addedResource);
            eventParams.put(ObservationConstants.ARGS_PARENT_ID, target.getId());
            eventParams.put(ObservationConstants.ARGS_PARENT_PATH, target.getPath());

            _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_CREATED, _currentUserProvider.getUser(), eventParams));
        }

        if (uncopiedResources.size() > 0)
        {
            result.put("uncopied-resources", uncopiedResources);
        }

        if (copiedResourceIds.size() > 0)
        {
            result.put("copied-resources", copiedResourceIds);
        }

        return result;
    }

    /**
     * Move objects
     * 
     * @param ids The list of identifiers for the objects to move
     * @param targetId The id of target to move into
     * @return The result map with a message key in case of an error or with the
     *         list of unmoved/moved objects
     * @throws RepositoryException If there is a repository error
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> moveObject(List<String> ids, String targetId) throws RepositoryException
    {
        assert ids != null;
        assert targetId != null;

        AmetysObject target = _resolver.resolveById(targetId);
        if (!(target instanceof JCRTraversableAmetysObject))
        {
            throw new IllegalClassException(JCRTraversableAmetysObject.class, target.getClass());
        }

        return moveObject(ids, (JCRTraversableAmetysObject) target);
    }

    /**
     * Move objects
     * 
     * @param ids The list of identifiers for the objects to move
     * @param targetNode The target to move into
     * @return The result map with a message key in case of an error or with the
     *         list of unmoved/moved objects
     * @throws RepositoryException If there is a repository error
     */
    public Map<String, Object> moveObject(List<String> ids, JCRTraversableAmetysObject targetNode) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        List<String> unmovedObjects = new ArrayList<>();
        List<String> noRightObjects = new ArrayList<>();
        List<String> movedObjectIds = new ArrayList<>();

        assert ids != null;
        assert targetNode != null;

        if (!checkLock(targetNode))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to move objet to '" + targetNode.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        for (String id : ids)
        {
            JCRAmetysObject object = (JCRAmetysObject) _resolver.resolveById(id);
            String addRightId = object instanceof Resource ? RIGHTS_RESOURCE_ADD : RIGHTS_COLLECTION_ADD;
            String deleteRightId = object instanceof Resource ? RIGHTS_RESOURCE_DELETE : RIGHTS_COLLECTION_DELETE;
            ResourceCollection oldParent = object.getParent();
            
            if (_rightManager.hasRight(_currentUserProvider.getUser(), addRightId, targetNode) != RightResult.RIGHT_ALLOW || _rightManager.hasRight(_currentUserProvider.getUser(), deleteRightId, oldParent) != RightResult.RIGHT_ALLOW)
            {
                getLogger().error("User '" + _currentUserProvider.getUser() + "' tried to move resource without convenient right [" + addRightId + "]");
                noRightObjects.add(object.getName());
            }
            else if (targetNode.hasChild(object.getName()))
            {
                getLogger().warn("The object '" + object.getName() + "' can not be moved : a object of same name already exists in the target collection");
                result.put("message", "already-exist");
                unmovedObjects.add(object.getName());
            }
            else
            {
                String oldResourcePath = object instanceof Resource ? ((Resource) object).getResourcePath() : ((ExplorerNode) object).getExplorerPath();

                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put("object.old.path", object.getPath());

                Session session = object.getNode().getSession();
                session.move(object.getNode().getPath(), targetNode.getNode().getPath() + "/" + object.getNode().getName());

                if (object instanceof Resource)
                {
                    eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, ((Resource) object).getResourcePath());
                    eventParams.put("resource.old.path", oldResourcePath);
                }
                else
                {
                    eventParams.put("explorer.old.path", oldResourcePath);
                }

                eventParams.put(ObservationConstants.ARGS_PATH, ((ExplorerNode) targetNode).getExplorerPath() + "/" + object.getName());
                eventParams.put(ObservationConstants.ARGS_PARENT_ID, targetNode.getId());
                eventParams.put(ObservationConstants.ARGS_ID, id);
                eventParams.put(ObservationConstants.ARGS_NAME, object.getName());

                session.save();
                movedObjectIds.add(object.getId());

                if (object instanceof Resource)
                {
                    eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, ((Resource) object).getResourcePath());
                    eventParams.put("resource.old.path", oldResourcePath);
                    _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_MOVED, _currentUserProvider.getUser(), eventParams));
                }
                else if (object instanceof ResourceCollection)
                {
                    _observationManager.notify(new Event(ObservationConstants.EVENT_COLLECTION_MOVED, _currentUserProvider.getUser(), eventParams));
                }
                else
                {
                    getLogger().warn("Object " + object.getId() + " of class '" + object.getClass().getName() + "' was moved. This type is unknown.");
                }
            }
        }

        if (!unmovedObjects.isEmpty())
        {
            result.put("unmoved-objects", unmovedObjects);
        }
        if (!noRightObjects.isEmpty())
        {
            result.put("noright-objects", noRightObjects);
        }
        if (!movedObjectIds.isEmpty())
        {
            result.put("moved-objects", movedObjectIds);
        }

        return result;
    }

    /**
     * Get the history versions of a resource
     * 
     * @param id the id of resource
     * @return The versions
     * @throws RepositoryException if an error occurred
     */
    // FIXME no read access is checked as resource tree does not filter resources on read access
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, Object>> resourceHistory(String id) throws RepositoryException
    {
        JCRResource resource = _resolver.resolveById(id);

        List<Map<String, Object>> versions = new ArrayList<>();

        List<VersionInformation> versionsInfo = new ArrayList<>();

        for (String revision : resource.getAllRevisions())
        {
            VersionInformation versionInformation = new VersionInformation(revision, resource.getRevisionTimestamp(revision));

            for (String label : resource.getLabels(revision))
            {
                versionInformation.addLabel(label);
            }

            versionsInfo.add(versionInformation);
        }

        // Sort by date descendant
        Collections.sort(versionsInfo, new Comparator<VersionInformation>()
        {
            public int compare(VersionInformation o1, VersionInformation o2)
            {
                try
                {
                    return -o1.getCreatedAt().compareTo(o2.getCreatedAt());
                }
                catch (RepositoryException e)
                {
                    throw new RuntimeException("Unable to retrieve a creation date", e);
                }
            }
        });

        for (VersionInformation versionInformation : versionsInfo)
        {
            Map<String, Object> version = _version2json(resource, versionInformation);
            versions.add(version);
        }

        return versions;
    }
    
    /**
     * Get the JSON representation of a version of a resource
     * @param resource the resource
     * @param versionInformation the version information
     * @return the version as JSON object
     * @throws RepositoryException if failed to get revision
     */
    protected Map<String, Object> _version2json(JCRResource resource, VersionInformation versionInformation) throws RepositoryException
    {
        Map<String, Object> version = new HashMap<>();

        for (String label : versionInformation.getLabels())
        {
            version.put(label, label);
        }

        version.put("rawName", versionInformation.getVersionRawName());
        version.put("name", versionInformation.getVersionName());
        version.put("createdAt", DateUtils.dateToString(versionInformation.getCreatedAt()));

        try
        {
            resource.switchToRevision(versionInformation.getVersionRawName());
            UserIdentity author = resource.getLastContributor();
            version.put("author", _userHelper.user2json(author));
        }
        catch (AmetysRepositoryException e)
        {
            // Do nothing. Can append with old version history (before 4.0)
        }
        
        return version;
    }

    /**
     * This action restores an old version of a {@link Resource}.
     * 
     * @param id the id of resource
     * @param versionName the name of version to restore
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public void restoreResource(String id, String versionName)
    {
        JCRResource resource = _resolver.resolveById(id);
        
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_ADD, resource.getParent()) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to restore file without convenient right [" + RIGHTS_RESOURCE_ADD + "]");
        }
        
        resource.restoreFromRevision(versionName);

        resource.setLastContributor(_currentUserProvider.getUser());
        resource.setLastModified(new Date());
        resource.saveChanges();
        resource.checkpoint();
    }

    /**
     * Determines if at least one resource with given names already exists
     * 
     * @param parentId the id of parent collection
     * @param names the names of the resources
     * @return true if a resource with same name exists
     */
    // FIXME Check of read access should be enought but read access is not necessary to add a resource
    @Callable (rights = {Callable.READ_ACCESS, RIGHTS_RESOURCE_ADD}, paramIndex = 0, rightContext = ResourceRightAssignmentContext.ID)
    public boolean resourcesExists(String parentId, List<String> names)
    {
        for (String name : names)
        {
            if (resourceExists(parentId, name))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Determines if a resource with given name already exists
     * 
     * @param parentId the id of parent collection
     * @param name the name of resource
     * @return true if a resource with same name exists
     */
    public boolean resourceExists(String parentId, String name)
    {
        return resourceExists((TraversableAmetysObject) _resolver.resolveById(parentId), name);
    }

    /**
     * Determines if a resource with given name already exists
     * 
     * @param parent the parent collection
     * @param name the name of resource
     * @return true if a resource with same name exists
     */
    public boolean resourceExists(TraversableAmetysObject parent, String name)
    {
        return parent.hasChild(name);
    }

    /**
     * Set the Dublin Core metadata of a {@link Resource}.
     * 
     * @param resourceId the id of resource
     * @param values the DC values
     * @throws ProcessingException if an error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public void setDCMetadata(String resourceId, Map<String, Object> values) throws ProcessingException
    {
        setDCMetadata((ModifiableResource) _resolver.resolveById(resourceId), values);
    }

    /**
     * Set the Dublin Core metadata of a {@link Resource}.
     * 
     * @param resource the resource
     * @param values the DC values
     */
    public void setDCMetadata(ModifiableResource resource, Map<String, Object> values)
    {
        String title = (String) values.get("dc_title");
        String creator = (String) values.get("dc_creator");
        Object subject = values.get("dc_subject");
        @SuppressWarnings("unchecked")
        List<String> subjects = subject instanceof List ? (List<String>) subject : Collections.emptyList();
        String description = (String) values.get("dc_description");
        String publisher = (String) values.get("dc_publisher");
        String contributor = (String) values.get("dc_contributor");
        String dateStr = (String) values.get("dc_date");
        String type = (String) values.get("dc_type");
        String source = (String) values.get("dc_source");
        String language = (String) values.get("dc_language");
        String relation = (String) values.get("dc_relation");
        String coverage = (String) values.get("dc_coverage");
        String rights = (String) values.get("dc_rights");

        try
        {
            if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_RESOURCE_EDIT_DC, resource) != RightResult.RIGHT_ALLOW)
            {
                throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit file DC without convenient right [" + RIGHTS_RESOURCE_EDIT_DC + "]");
            }
            
            _updateTitleIfNeeded(resource, title);
            _updateCreatorIfNeeded(resource, creator);
            _updateDateIfNeeded(resource, dateStr);
            _updateDCSubjectIfNeeded(resource, subjects);
            _updateDescriptionIfNeeded(resource, description);
            _updatePublisherIfNeeded(resource, publisher);
            _updateContributorIfNeeded(resource, contributor);
            _updateTypeIfNeeded(resource, type);
            _updateSourceIfNeeded(resource, source);
            _updateLanguageIfNeeded(resource, language);
            _updateRelationIfNeeded(resource, relation);
            _updateCoverageIfNeeded(resource, coverage);
            _updateRightsIfNeeded(resource, rights);

            _saveChangesIfNeeded(resource);

        }
        catch (AmetysRepositoryException e)
        {
            String errorMsg = String.format("Exception while trying to set Dublin Core metadata on resource %s", resource.getId());
            getLogger().error(errorMsg, e);
            throw new AmetysRepositoryException(errorMsg, e);
        }
    }

    private void _updateTitleIfNeeded(ModifiableResource resource, String title)
    {
        if (!Objects.equals(title, resource.getDCTitle()))
        {
            resource.setDCTitle(title);
        }
    }

    private void _updateCreatorIfNeeded(ModifiableResource resource, String creator)
    {
        if (!Objects.equals(creator, resource.getDCCreator()))
        {
            resource.setDCCreator(creator);
        }
    }

    private void _updateDateIfNeeded(ModifiableResource resource, String dateStr)
    {
        Date date = null;
        if (StringUtils.isNotEmpty(dateStr))
        {
            date = DateUtils.parse(dateStr);
        }

        if (!Objects.equals(date, resource.getDCDate()))
        {
            resource.setDCDate(date);
        }
    }

    private void _updateRightsIfNeeded(ModifiableResource resource, String rights)
    {
        if (!Objects.equals(rights, resource.getDCRights()))
        {
            resource.setDCRights(rights);
        }
    }

    private void _updateCoverageIfNeeded(ModifiableResource resource, String coverage)
    {
        if (!Objects.equals(coverage, resource.getDCCoverage()))
        {
            resource.setDCCoverage(coverage);
        }
    }

    private void _updateRelationIfNeeded(ModifiableResource resource, String relation)
    {
        if (!Objects.equals(relation, resource.getDCRelation()))
        {
            resource.setDCRelation(relation);
        }
    }

    private void _updateLanguageIfNeeded(ModifiableResource resource, String language)
    {
        if (!Objects.equals(language, resource.getDCLanguage()))
        {
            resource.setDCLanguage(language);
        }
    }

    private void _updateSourceIfNeeded(ModifiableResource resource, String source)
    {
        if (!Objects.equals(source, resource.getDCSource()))
        {
            resource.setDCSource(source);
        }
    }

    private void _updateTypeIfNeeded(ModifiableResource resource, String type)
    {
        if (!Objects.equals(type, resource.getDCType()))
        {
            resource.setDCType(type);
        }
    }

    private void _updateContributorIfNeeded(ModifiableResource resource, String contributor)
    {
        if (!Objects.equals(contributor, resource.getDCContributor()))
        {
            resource.setDCContributor(contributor);
        }
    }

    private void _updatePublisherIfNeeded(ModifiableResource resource, String publisher)
    {
        if (!Objects.equals(publisher, resource.getDCPublisher()))
        {
            resource.setDCPublisher(publisher);
        }
    }

    private void _updateDescriptionIfNeeded(ModifiableResource resource, String description)
    {
        if (!Objects.equals(description, resource.getDCDescription()))
        {
            resource.setDCDescription(description);
        }
    }

    private void _updateDCSubjectIfNeeded(ModifiableResource resource, List<String> subjects)
    {
        String[] trimSubject = subjects.stream().map(StringUtils::trim).toArray(String[]::new);
        if (!Objects.deepEquals(trimSubject, resource.getDCSubject()))
        {
            resource.setDCSubject(trimSubject);
        }
    }
    
    private void _saveChangesIfNeeded(ModifiableResource resource)
    {
        if (resource.needsSave())
        {
            resource.setLastContributor(_currentUserProvider.getUser());
            resource.setLastModified(new Date());

            ModifiableResourceCollection parent = resource.getParent();
            parent.saveChanges();

            if (resource instanceof VersionableAmetysObject)
            {
                // Create first version
                ((VersionableAmetysObject) resource).checkpoint();
            }

            // Notify listeners of resource update.
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_ID, resource.getId());
            eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
            eventParams.put(ObservationConstants.ARGS_NAME, resource.getName());
            eventParams.put(ObservationConstants.ARGS_PATH, resource.getPath());

            _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_UPDATED, _currentUserProvider.getUser(), eventParams));
        }
    }

    /**
     * Get the DublinCore metadata of a {@link DublinCoreAwareAmetysObject}
     * 
     * @param id the id of resource
     * @return the DC metadata
     */
    @Callable (rights = {Callable.READ_ACCESS, RIGHTS_RESOURCE_EDIT_DC}, paramIndex = 0, rightContext = ResourceRightAssignmentContext.ID)
    public Map<String, Object> getDCMetadata(String id)
    {
        AmetysObject object = _resolver.resolveById(id);
        if (object instanceof DublinCoreAwareAmetysObject)
        {
            DublinCoreAwareAmetysObject dcObject = (DublinCoreAwareAmetysObject) object;

            Map<String, Object> metadata = new HashMap<>();

            metadata.put("id", id);

            Map<String, Object> values = new HashMap<>();

            values.put("dc_title", dcObject.getDCTitle());
            values.put("dc_creator", dcObject.getDCCreator());
            values.put("dc_subject", dcObject.getDCSubject());
            values.put("dc_description", dcObject.getDCDescription());
            values.put("dc_type", dcObject.getDCType());
            values.put("dc_publisher", dcObject.getDCPublisher());
            values.put("dc_contributor", dcObject.getDCContributor());
            values.put("dc_date", DateUtils.dateToString(dcObject.getDCDate()));
            values.put("dc_format", dcObject.getDCFormat());
            values.put("dc_identifier", dcObject.getDCIdentifier());
            values.put("dc_source", dcObject.getDCSource());
            values.put("dc_language", dcObject.getDCLanguage());
            values.put("dc_relation", dcObject.getDCRelation());
            values.put("dc_coverage", dcObject.getDCCoverage());
            values.put("dc_rights", dcObject.getDCRights());

            metadata.put("values", values);
            return metadata;
        }
        else
        {
            throw new IllegalArgumentException("Object of id " + id + " is not Dublin Core aware.");
        }
    }

    /**
     * Creates a new CMIS folder (see {@link CMISRootResourcesCollection})
     * 
     * @param parentId the id of parent folder
     * @param originalName the original name if CMIS folder
     * @param url The url of CMIS repository
     * @param login The user's login to access CMIS repository
     * @param password The user's password to access CMIS repository
     * @param repoId The id of CMIS repository
     * @param mountPoint The mount point in the server
     * @param renameIfExists true to automatically renamed CMIS folder if
     *            requested name already exists
     * @return the result map with id of created node
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addCMISCollection(String parentId, String originalName, String url, String login, String password, String repoId, String mountPoint, boolean renameIfExists)
    {
        Map<String, Object> result = new HashMap<>();

        AmetysObject object = _resolver.resolveById(parentId);
        if (!(object instanceof ModifiableResourceCollection))
        {
            throw new IllegalClassException(ModifiableResourceCollection.class, object.getClass());
        }

        ModifiableResourceCollection collection = (ModifiableResourceCollection) object;
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_CMIS_ADD, collection) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to add CMIS collection without convenient right [" + RIGHTS_COLLECTION_CMIS_ADD + "]");
        }
        
        if (!checkLock(collection))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify collection '" + object.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }

        if (!renameIfExists && collection.hasChild(originalName))
        {
            getLogger().warn("The object '" + object.getName() + "' can not be renamed in '" + originalName + "' : a object of same name already exists.");
            result.put("message", "already-exist");
            return result;
        }

        int index = 1;
        String name = originalName;
        while (collection.hasChild(name))
        {
            name = originalName + " (" + index + ")";
            index++;
        }

        CMISRootResourcesCollection child = collection.createChild(name, CMISTreeFactory.CMIS_ROOT_COLLECTION_NODETYPE);

        child.setRepositoryUrl(url);
        child.setRepositoryId(repoId);
        child.setMountPoint(mountPoint);
        child.setUser(login);
        child.setPassword(password);

        collection.saveChanges();

        result.put("id", child.getId());
        result.put("parentID", parentId);
        result.put("name", name);

        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, child.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, object.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, child.getName());
        eventParams.put(ObservationConstants.ARGS_PATH, child.getPath());

        _observationManager.notify(new Event(ObservationConstants.EVENT_COLLECTION_CREATED, _currentUserProvider.getUser(), eventParams));

        return result;
    }

    /**
     * Edits a CMIS folder (see {@link CMISRootResourcesCollection})
     * 
     * @param id the id of CMIS folder
     * @param url The url of CMIS repository
     * @param login The user's login to access CMIS repository
     * @param password The user's password to access CMIS repository
     * @param repoId The id of CMIS repository
     * @param mountPoint The mount point to use for the repository
     * @return the result map with id of edited node
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editCMISCollection(String id, String url, String login, String password, String repoId, String mountPoint)
    {
        Map<String, Object> result = new HashMap<>();

        CMISRootResourcesCollection object = _resolver.resolveById(id);

        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_CMIS_ADD, object) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit CMIS collection without convenient right [" + RIGHTS_COLLECTION_CMIS_ADD + "]");
        }

        object.setRepositoryUrl(url);
        object.setRepositoryId(repoId);
        object.setMountPoint(mountPoint);
        object.setUser(login);
        object.setPassword(password);

        object.saveChanges();

        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, object.getId());

        _observationManager.notify(new Event(ObservationConstants.EVENT_CMIS_COLLECTION_UPDATED, _currentUserProvider.getUser(), eventParams));

        result.put("id", object.getId());

        return result;
    }

    /**
     * Determines if a object is a {@link CMISRootResourcesCollection}
     * 
     * @param id The id of object
     * @return true if it is a CMIS root folder
     */
    @Callable (rights = {Callable.READ_ACCESS, RIGHTS_COLLECTION_CMIS_ADD}, paramIndex = 0, rightContext = ResourceRightAssignmentContext.ID)
    public boolean isCMISCollection(String id)
    {
        ResourceCollection collection = _resolver.resolveById(id);
        return collection instanceof CMISRootResourcesCollection;
    }

    /**
     * Get the CMIS properties of collection
     * 
     * @param id The id of CMIS collection
     * @return the CMIS properties to access CMIS repository
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> getCMISProperties(String id)
    {
        Map<String, String> properties = new HashMap<>();

        CMISRootResourcesCollection cmis = _resolver.resolveById(id);
        
        ResourceCollection parent = cmis.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COLLECTION_CMIS_ADD, parent) != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to getCMIS properties without convenient right [" + RIGHTS_COLLECTION_ADD + "]");
        }
        
        properties.put("id", cmis.getId());
        properties.put("name", cmis.getName());
        properties.put("url", cmis.getRepositoryUrl());
        properties.put("login", cmis.getUser());
        properties.put("password", cmis.getPassword());
        properties.put("repoId", cmis.getRepositoryId());
        properties.put("mountPoint", cmis.getMountPoint());

        return properties;
    }

    /**
     * Updates resource input stream and metadata
     * 
     * @param resource The resource
     * @param is The resource input stream
     * @param fileName The file name
     */
    public void updateResource(ModifiableResource resource, InputStream is, String fileName)
    {
        UserIdentity author = _currentUserProvider.getUser();

        String mimeType = _cocoonContext.getMimeType(fileName.toLowerCase());
        mimeType = mimeType == null ? "application/unknown" : mimeType;

        resource.setData(is, mimeType, new Date(), author);
        resource.setLastModified(new Date());

        extractResourceMetadata(resource, mimeType);
    }

    /**
     * Extract the resource's metadata and populate the object accordingly.
     * 
     * @param resource the resource to populate.
     * @param mimeType the resource MIME type.
     */
    public void extractResourceMetadata(ModifiableResource resource, String mimeType)
    {
        try (InputStream is = resource.getInputStream())
        {
            Metadata metadata = new Metadata();

            try (Reader reader = _tikaProvider.getTika().parse(is, metadata))
            {
                IOUtils.copy(reader, OutputStream.nullOutputStream(), StandardCharsets.UTF_8);

                Collection<ResourceMetadataPopulator> populators = _metadataPopulatorEP.getPopulators(mimeType);
                for (ResourceMetadataPopulator populator : populators)
                {
                    populator.populate(resource, metadata);
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error populating the metadata of resource " + resource.getId(), e);
        }
    }

    /**
     * Creates a new version
     * 
     * @param resource the resource
     */
    public void checkpoint(ModifiableResource resource)
    {
        if (resource instanceof VersionableAmetysObject)
        {
            // Create first version
            ((VersionableAmetysObject) resource).checkpoint();
        }
    }

    /**
     * Creates a {@link ModifiableResource} under current
     * {@link ModifiableResourceCollection}.
     * 
     * @param collection the parent {@link ModifiableResourceCollection}
     * @param name the name of the child resource
     * @return the new resource created.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryIntegrityViolationException if an object with the same
     *             name already exists and same name siblings is not allowed.
     */
    public ModifiableResource createResource(ModifiableResourceCollection collection, String name)
    {
        return collection.createChild(name, collection.getResourceType());
    }

    /**
     * Bean for version information
     *
     */
    protected static class VersionInformation
    {
        private String _rawName;

        private String _name;

        private Date _creationDate;

        private Set<String> _labels = new HashSet<>();

        /**
         * Creates a {@link VersionInformation}.
         * 
         * @param rawName the revision name.
         * @param creationDate the revision creation date.
         * @throws RepositoryException if an error occurs.
         */
        public VersionInformation(String rawName, Date creationDate) throws RepositoryException
        {
            _creationDate = creationDate;
            _rawName = rawName;
            // 1.0 > v0, 1.1 > v1, ...
            _name = String.valueOf(Integer.parseInt(_rawName.substring(2)) + 1);
        }

        /**
         * Retrieves the version name.
         * 
         * @return the version name.
         */
        public String getVersionName()
        {
            return _name;
        }

        /**
         * Retrieves the version raw name.
         * 
         * @return the version raw name.
         */
        public String getVersionRawName()
        {
            return _rawName;
        }

        /**
         * Retrieves the creation date.
         * 
         * @return the creation date.
         * @throws RepositoryException if an error occurs.
         */
        public Date getCreatedAt() throws RepositoryException
        {
            return _creationDate;
        }

        /**
         * Retrieves the labels associated with this version.
         * 
         * @return the labels.
         */
        public Set<String> getLabels()
        {
            return _labels;
        }

        /**
         * Add a label to this version.
         * 
         * @param label the label.
         */
        public void addLabel(String label)
        {
            _labels.add(label);
        }
    }
}
