/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.explorer.resources.dom;

import java.time.ZoneId;
import java.util.Date;
import java.util.Map;

import org.w3c.dom.Element;

import org.ametys.core.util.DateUtils;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.plugins.explorer.resources.Resource;

/**
 * DOM {@link Element} wrapping a {@link Resource}.
 */
public class ResourceElement extends AbstractResourceElement<Resource>
{
    /**
     * Constructor.
     * @param resource the wrapped resource.
     * @param parent the parent collection.
     */
    public ResourceElement(Resource resource, ResourceCollectionElement parent)
    {
        super(resource, parent);
    }
    
    @Override
    public String getTagName()
    {
        return "resource";
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> atts = super._lookupAttributes();
        atts.put("name", new AmetysAttribute("name", "name", null, _object.getName(), this));
        atts.put("id", new AmetysAttribute("id", "id", null, _object.getId(), this));
        atts.put("length", new AmetysAttribute("length", "length", null, String.valueOf(_object.getLength()), this));
        
        Date lastModified = _object.getLastModified();
        if (lastModified != null)
        {
            String lastModifiedStr = DateUtils.getISODateTimeFormatter().format(lastModified.toInstant().atZone(ZoneId.systemDefault()));
            
            atts.put("last-modified", new AmetysAttribute("last-modified", "last-modified", null, lastModifiedStr, this));
        }
        
        return atts;
    }
}
