/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.generators;

import java.io.IOException;
import java.util.Comparator;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.core.util.StringUtils.AlphanumComparator;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;

/**
 * Generates a subtree of {@link ExplorerNode}.<br>The subnodes are SAXed to a depth of 0 by default.
 * <ul>
 *   <li>Depth of -1 means generate all</li>
 *   <li>0 means only the given node and its children (resources and folders)</li>
 *   <li>1 means the given node, its children and their children, and so on.</li>
 * </ul>
 */
public class ResourcesExplorerGenerator extends ServiceableGenerator
{
    /** Constant for resource collection */
    public static final String RESOURCE_COLLECTION = "collection";
    /** Constant for resource */
    public static final String RESOURCE = "resource";
    
    /** The resolver for ametys object */
    protected AmetysObjectResolver _resolver;
    /** The user manager */
    protected UserManager _userManager;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        
        _resolver = (AmetysObjectResolver) sManager.lookup(AmetysObjectResolver.ROLE);
        _userManager = (UserManager) sManager.lookup(UserManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = parameters.getParameter("node", request.getParameter("node"));
        int depth = parameters.getParameterAsInteger("depth", 0);
        if (depth == -1)
        {
            // -1 means no limit
            depth = Integer.MAX_VALUE;
        }
        
        String[] allowedExtensions = request.getParameterValues("allowedExtensions");
        
        ExplorerNode node = _resolver.resolveById(id);
        
        contentHandler.startDocument();
        
        long t0 = System.currentTimeMillis();
        
        XMLUtils.startElement(contentHandler, "Nodes");
        saxCollection(node, depth, allowedExtensions);
        XMLUtils.endElement(contentHandler, "Nodes");
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("SAXed collection " + node.getExplorerPath() + " in " + (System.currentTimeMillis() - t0) + " ms");
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX a {@link ExplorerNode}
     * @param node The node to sax
     * @throws SAXException If an error occurred while saxing
     */
    protected void saxCollection (ExplorerNode node) throws SAXException
    {
        saxCollection(node, 0, null);
    }

    /**
     * SAX a {@link ExplorerNode}
     * @param node The node to sax
     * @param saxNode True to wrap the collection with an XML node representing the current node.
     * @throws SAXException If an error occurred while saxing
     */
    protected void saxCollection (ExplorerNode node, boolean saxNode) throws SAXException
    {
        saxCollection(node, 0, saxNode, "Node");
    }
    
    /**
     * SAX a {@link ExplorerNode}
     * @param node The node to sax
     * @param saxNode True to wrap the collection with an XML node representing the current node.
     * @param nodeTag The tag to use
     * @throws SAXException If an error occurred while saxing
     */
    protected void saxCollection (ExplorerNode node, boolean saxNode, String nodeTag) throws SAXException
    {
        saxCollection(node, 0, saxNode, nodeTag);
    }
    
    /**
     * SAX a {@link ExplorerNode}
     * @param node The node to sax
     * @param depth The recursive depth to sax
     * @param saxNode True to wrap the collection with an XML node representing the current node.
     * @param nodeTag The tag to use
     * @throws SAXException If an error occurred while saxing
     */
    protected void saxCollection(ExplorerNode node, int depth, boolean saxNode, String nodeTag) throws SAXException
    {
        if (saxNode)
        {
            AttributesImpl childAtts = getExplorerNodeAttributes(node);
            
            XMLUtils.startElement(contentHandler, nodeTag, childAtts);
            saxCollection(node, depth, null);
            XMLUtils.endElement(contentHandler, nodeTag);
        }

    }
    
    /**
     * SAX a {@link ExplorerNode}
     * @param node The node to sax
     * @param depth The recursive depth to sax
     * @param allowedExtensions The allowed file extensions (lower-case). Can be null or empty to not filter on file extensions
     * @throws SAXException If an error occurred while saxing
     */
    protected void saxCollection (ExplorerNode node, int depth, String[] allowedExtensions) throws SAXException
    {
        // Traverse the child nodes depth >= 0 (we're not in the last level).
        if (depth >= 0)
        {
            AmetysObjectIterable<AmetysObject> children = ((TraversableAmetysObject) node).getChildren();
            
            // First sax folder on alphanum order (case insensitive)
            children
                .stream()
                .filter(ExplorerNode.class::isInstance)
                .map(ExplorerNode.class::cast)
                .sorted(Comparator.comparing(ExplorerNode::getName, new AlphanumComparator()))
                .forEach(LambdaUtils.wrapConsumer(n -> {
                    saxExplorerNode(n, depth);
                }));
            
            // Then sax files on alphanum order (case insensitive)
            children
                .stream()
                .filter(Resource.class::isInstance)
                .map(Resource.class::cast)
                .filter(r -> _matchFilter(r, allowedExtensions))
                .sorted(Comparator.comparing(Resource::getName, new AlphanumComparator()))
                .forEach(LambdaUtils.wrapConsumer(r -> {
                    saxResource(r);
                }));
        }
    }
    
    /**
     * Determines if the resource matches the allowed extensions
     * @param resource The resource 
     * @param allowedExtensions allowed file extensions
     * @return true if the allowed extensions are empty or null, or if the resource's name matches the allowed extensions
     */
    protected boolean _matchFilter(Resource resource, String[] allowedExtensions)
    {
        if (allowedExtensions == null || allowedExtensions.length == 0)
        {
            // No filter
            return true;
        }
        
        String filename = resource.getName();
        String extension = filename.lastIndexOf(".") > 0 ? filename.substring(filename.lastIndexOf(".") + 1) : "";
        
        return ArrayUtils.contains(allowedExtensions, extension.toLowerCase());
    }

    /**
     * SAX a {@link ExplorerNode}
     * @param node The explorer node to SAX.
     * @param depth The recursive depth to sax
     * @throws SAXException If an error occurred while SAXing
     */
    protected void saxExplorerNode(ExplorerNode node, int depth) throws SAXException
    {
        saxExplorerNode(node, depth, null);
    }
    
    /**
     * SAX a {@link ExplorerNode}
     * @param node The explorer node to SAX.
     * @param depth The recursive depth to sax
     * @param allowedExtensions allowed file extensions
     * @throws SAXException If an error occurred while SAXing
     */
    protected void saxExplorerNode(ExplorerNode node, int depth, String[] allowedExtensions) throws SAXException
    {
        AttributesImpl childAtts = getExplorerNodeAttributes(node);
        
        XMLUtils.startElement(contentHandler, "Node", childAtts);
        saxCollection(node, depth - 1, allowedExtensions);
        XMLUtils.endElement(contentHandler, "Node");
    }
    
    /**
     * Returns the attributes corresponding to an Explorer node.
     * @param node The explorer node to SAX.
     * @return the attributes
     */
    protected AttributesImpl getExplorerNodeAttributes(ExplorerNode node)
    {
        AttributesImpl childAtts = new AttributesImpl();
        
        childAtts.addCDATAAttribute("id", node.getId());
        childAtts.addCDATAAttribute("name", node.getName());
        
        String iconCls = node.getIconCls();
        if (iconCls != null)
        {
            childAtts.addCDATAAttribute("iconCls", iconCls);
        }
        childAtts.addCDATAAttribute("applicationId", node.getApplicationId());
        
        String relPath = node.getExplorerPath();
        childAtts.addCDATAAttribute("path", _getNodePath(relPath));
        childAtts.addCDATAAttribute("type", RESOURCE_COLLECTION);
        
        boolean hasResources = false;
        if (node instanceof ResourceCollection)
        {
            hasResources = ((ResourceCollection) node).hasChildResources();
        }
        boolean hasChildNodes = node.hasChildExplorerNodes();
        
        if (hasChildNodes)
        {
            childAtts.addCDATAAttribute("hasChildNodes", "true");
        }
        
        if (hasResources)
        {
            childAtts.addCDATAAttribute("hasResources", "true");
        }
        
        if (node instanceof ModifiableAmetysObject)
        {
            childAtts.addCDATAAttribute("isModifiable", "true");
        }
        
        if (node instanceof ModifiableExplorerNode)
        {
            childAtts.addCDATAAttribute("canCreateChild", "true");
        }
        
        return childAtts;
    }
    
    /**
     * SAX a {@link Resource}
     * @param resource The resource to SAX
     * @throws SAXException If an error occurred while SAXing
     */
    protected void saxResource (Resource resource) throws SAXException
    {
        AttributesImpl childAtts = getResourceAttributes(resource);
        
        XMLUtils.createElement(contentHandler, "Node", childAtts);
    }
    
    /**
     * Returns the attributes corresponding to a Resource.
     * @param resource The resource to SAX
     * @return the attributes
     */
    protected AttributesImpl getResourceAttributes(Resource resource)
    {
        UserIdentity userIdentity = resource.getCreator();
        User user = _userManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
        String name = user == null ? userIdentity.getLogin() : user.getFullName() + " (" + userIdentity.getLogin() + ")"; 
        
        AttributesImpl childAtts = new AttributesImpl();
        childAtts.addCDATAAttribute("id", resource.getId());
        String iconCls = getResourcesIconCls(resource);
        if (iconCls != null)
        {
            childAtts.addCDATAAttribute("iconCls", getResourcesIconCls(resource));
        }
        childAtts.addCDATAAttribute("name", resource.getName());
        childAtts.addCDATAAttribute("mimetype", resource.getMimeType());
        childAtts.addCDATAAttribute("lastModified", DateUtils.dateToString(resource.getLastModified()));
        childAtts.addCDATAAttribute("size", String.valueOf(resource.getLength()));
        childAtts.addCDATAAttribute("author", name);
        childAtts.addCDATAAttribute("keywords", resource.getKeywordsAsString());
        childAtts.addCDATAAttribute("type", RESOURCE);
        
        if (resource instanceof ModifiableAmetysObject)
        {
            childAtts.addCDATAAttribute("isModifiable", "true");
        }
        
        getAdditionalAttributes(childAtts, resource);
        
        String relPath = resource.getResourcePath();
        childAtts.addCDATAAttribute("path", _getNodePath(relPath));
        
        return childAtts;
    }

    /**
     * Remove elements from the path to make it compatible with ExplorerTree data model
     * @param relPath the path to parse
     * @return the node's path
     */
    protected String _getNodePath(String relPath)
    {
        return relPath.startsWith("/") ? relPath.substring(1) : relPath;
    }

    /**
     * CSS suffix class name getter for the icon resource.
     * @param resource The resource
     * @return The suffix of the css class for the icon of this resource.
     */
    protected String getResourcesIconCls(Resource resource)
    {
        // let the js do its job
        return null;
    }
    
    /**
     * Get the additional attributes
     * @param attrs The attributes
     * @param resource The resource
     */
    protected void getAdditionalAttributes (AttributesImpl attrs, Resource resource)
    {
        // Nothing
    }
}
