/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.generators;

import java.io.IOException;

import javax.jcr.RepositoryException;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.util.ISO9075;
import org.xml.sax.SAXException;

import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.LogicalOperator;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;

/**
 * Search resources
 *
 */
public class SearchGenerator extends ResourcesExplorerGenerator
{
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = request.getParameter("id");
        ExplorerNode node = _resolver.resolveById(id);
        
        Expression metaExpr = getMetadataExpression (request);
        
        String predicat = metaExpr != null ? metaExpr.build() : null;
        
        // File name
        String name = request.getParameter("name");
        if (StringUtils.isNotEmpty(name))
        {
            String namePredicate = "jcr:like(fn:lower-case(fn:name()),'%" + name.toLowerCase() + "%')";
            predicat = predicat == null ? namePredicate : predicat + LogicalOperator.OR.toString() + namePredicate;
        }
        
        // Full text
        String fulltext = request.getParameter("fulltext");
        if (StringUtils.isNotEmpty(fulltext))
        {
            String xpath = "jcr:contains(" + JcrConstants.JCR_CONTENT + ", '*" + fulltext + "*')";
            predicat = predicat == null ? xpath : predicat + LogicalOperator.OR.toString() + xpath;
        }
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "resources");
        
        try
        {
            if (node instanceof JCRAmetysObject)
            {
                String jcrPath = ((JCRAmetysObject) node).getNode().getPath();
                String relativeJcrPath = jcrPath.substring(AmetysObjectResolver.ROOT_REPO.length() + 1);
                String xpathQuery = "/" + ISO9075.encodePath(relativeJcrPath) + "//element(*, ametys:resource)" + (predicat != null ? "[" + predicat + "]" : "");
                
                try (AmetysObjectIterable<Resource> it = _resolver.query(xpathQuery))
                {
                    for (Resource resource : it)
                    {
                        saxResource(resource);
                    }
                }
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("Error getting the JCR path of the explorer node of id " + id, e);
            throw new ProcessingException("Error getting the JCR path of the explorer node of id " + id, e);
        }
        
        XMLUtils.endElement(contentHandler, "resources");
        contentHandler.endDocument();
    }
    
    /**
     * Get the {@link Expression}
     * @param request The request
     * @return the {@link Expression}
     */
    protected Expression getMetadataExpression (Request request)
    {
        Expression expr = null;
        
        String keywords = request.getParameter("keywords");
        if (StringUtils.isNotEmpty(keywords))
        {
            String[] words = StringUtils.split(keywords);
            for (String word : words)
            {
                ExpressionContext context = ExpressionContext.newInstance().withUnversioned(true);
                Expression subjectExpr = new StringExpression("dc/dc_subject", Operator.WD, word, context);
                Expression descExpr = new StringExpression("dc/dc_description",  Operator.WD, word, context);
                
                expr = expr == null ? new OrExpression(subjectExpr, descExpr) : new OrExpression(expr, subjectExpr, descExpr);
            }
        }
        
        String author = request.getParameter("author");
        if (StringUtils.isNotEmpty(author))
        {
            Expression authorExpr = new UserExpression("author", Operator.EQ, author, false, true);
            expr = expr == null ? authorExpr : new OrExpression(expr, authorExpr);
        }
        
        return expr;
    }

    @Override
    protected AttributesImpl getResourceAttributes(Resource resource)
    {
        AttributesImpl atts = super.getResourceAttributes(resource);
        
        AmetysObject parent = resource.getParent();
        AmetysObject object = resource;
        while (parent instanceof ResourceCollection)
        {
            object = parent;
            parent = object.getParent();
        }
        
        atts.addCDATAAttribute("rootId", object.getId());
        
        return atts;
    }
}
