/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.generators;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.serialization.ZipArchiveSerializer;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Generates a ZIP archive for the selected resources
 *
 */
public class ZipArchiveGenerator extends ServiceableGenerator
{
    private AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = request.getParameter("id");
        String[] files = request.getParameterValues("file");
        boolean ignoreFolder = parameters.getParameterAsBoolean("ignoreFolder", false);
        
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("zip", ZipArchiveSerializer.ZIP_NAMESPACE);
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");

        if (files != null)
        {
            // Export the selected files (should have same parent)
            for (String fileId : files)
            {
                AmetysObject ao = _resolver.resolveById(fileId);
               
                if (!ignoreFolder && ao instanceof ResourceCollection)
                {
                    _saxResourceCollectionEntry ((ResourceCollection) ao, _getParentPath(ao), ignoreFolder);
                }
                else if (ao instanceof Resource)
                {
                    _saxResourceEntry ((Resource) ao, _getParentPath(ao));
                }
            }
        }
        else
        {
            ResourceCollection collection = _resolver.resolveById(id);
            _saxResourceCollectionEntry (collection, _getParentPath(collection), ignoreFolder);
        }
        
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        contentHandler.endPrefixMapping("zip");
        contentHandler.endDocument();
    }
    
    private String _getParentPath(AmetysObject ao)
    {
        String parentPath = "";
        if (ao.getParent() instanceof ExplorerNode)
        {
            parentPath = ((ExplorerNode) ao.getParent()).getExplorerPath();
        }
        
        return parentPath;
    }
    
    private void _saxResourceCollectionEntry (ResourceCollection collection, String fromPath, boolean ignoreFolder) throws SAXException
    {
        AmetysObjectIterable<AmetysObject> children = collection.getChildren();

        for (AmetysObject child : children)
        {
            if (child instanceof Resource)
            {
                Resource resource = (Resource) child;
                _saxResourceEntry(resource, fromPath);
            }
            else if (!ignoreFolder && child instanceof ResourceCollection)
            {
                _saxResourceCollectionEntry((ResourceCollection) child, fromPath, ignoreFolder);
            }
        }
    }
    
    private void _saxResourceEntry(Resource resource, String fromPath) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("name", resource.getResourcePath().substring(fromPath.length() + 1));
        attrs.addCDATAAttribute("src", "cocoon:/resource?id=" + URIUtils.encodeParameter(resource.getId()));
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry", attrs);
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry");
    }

}
