/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.jcr;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Set;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.version.VersionHistory;

import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CopiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.dublincore.DCMITypes;
import org.ametys.plugins.repository.jcr.DefaultLockableAmetysObject;
import org.ametys.plugins.repository.jcr.DublinCoreHelper;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.jcr.NameHelper.NameComputationMode;
import org.ametys.plugins.repository.jcr.NodeHelper;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.tag.TaggableAmetysObjectHelper;
import org.ametys.plugins.repository.trash.TrashElement;
import org.ametys.plugins.repository.trash.TrashableAmetysObject;
import org.ametys.plugins.repository.trash.UnknownParentException;

/**
 * Default implementation of an {@link Resource}, backed by a JCR node.<br>
 * @param <F> the actual type of factory.
 */
public class JCRResource<F extends JCRResourceFactory> extends DefaultLockableAmetysObject<F> implements ModifiableResource, CopiableAmetysObject, TrashableAmetysObject
{
    /** The name of node holding the creator */
    public static final String CREATOR_NODE_NAME = "creator";
    /** Constants for lastModified Metadata */
    public static final String CREATION_DATE = "creationDate";
    /** The name of node holding the last contributor */
    public static final String CONTRIBUTOR_NODE_NAME = "contributor";
    
    /**
     * Creates an {@link JCRResource}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public JCRResource(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }
    
    @Override
    public void setData(InputStream stream, String mimeType, Date lastModified, UserIdentity author)
    {
        Node fileNode = getNode();
        
        try
        {
            setLastContributor(author);
            
            Node resourceNode = null;
            
            if (fileNode.hasNode(JcrConstants.JCR_CONTENT))
            {
                // Already exists
                resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
            }
            else
            {
                resourceNode = fileNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_RESOURCE);
                setCreator(author);
                setCreationDate(new Date());
            }
            
            GregorianCalendar gc = new GregorianCalendar();
            gc.setTime(lastModified);
            resourceNode.setProperty(JcrConstants.JCR_LASTMODIFIED, gc);
            
            resourceNode.setProperty(JcrConstants.JCR_MIMETYPE, mimeType);
            
            Binary binary = resourceNode.getSession().getValueFactory().createBinary(stream);
            resourceNode.setProperty(JcrConstants.JCR_DATA, binary);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set data for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public void setLastModified(Date lastModified)
    {
        Node fileNode = getNode();
        
        try
        {
            Node resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
            
            GregorianCalendar gc = new GregorianCalendar();
            gc.setTime(lastModified);
            resourceNode.setProperty(JcrConstants.JCR_LASTMODIFIED, gc);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set lastmodified for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public void setKeywords(String keywords)
    {
        String[] words = StringUtils.stripAll(StringUtils.split(keywords, ','));
        
        String[] trimWords = new String[words.length];
        for (int i = 0; i < words.length; i++)
        {
            trimWords[i] = words[i].trim();
        }
        
        Node fileNode = getNode();
        try
        {
            fileNode.setProperty("ametys:keywords", trimWords);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set keywords for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public void setKeywords(String[] keywords)
    {
        Node fileNode = getNode();
        try
        {
            fileNode.setProperty("ametys:keywords", keywords);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set keywords for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public void setMimeType(String mimeType)
    {
        Node fileNode = getNode();
        
        try
        {
            Node resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
            
            resourceNode.setProperty(JcrConstants.JCR_MIMETYPE, mimeType);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set mimetype for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public void setCreator(UserIdentity author)
    {
        try
        {
            Node authorNode = null;
            if (getNode().hasNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATOR_NODE_NAME))
            {
                authorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATOR_NODE_NAME);
            }
            else
            {
                authorNode = getNode().addNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATOR_NODE_NAME, RepositoryConstants.USER_NODETYPE);
            }
            authorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", author.getLogin());
            authorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", author.getPopulationId());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set creator for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public InputStream getInputStream () throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            Node resourceNode = null;
            
            if (fileNode.hasNode(JcrConstants.JCR_CONTENT))
            {
                // Already exists
                resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
                return resourceNode.getProperty(JcrConstants.JCR_DATA).getBinary().getStream();
            }
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get inputstream for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public String getMimeType ()  throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            Node resourceNode = null;
            
            if (fileNode.hasNode(JcrConstants.JCR_CONTENT))
            {
                // Already exists
                resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
                return resourceNode.getProperty(JcrConstants.JCR_MIMETYPE).getString();
            }
            
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get mimetype for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public Date getLastModified () throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            Node resourceNode = null;
            
            if (fileNode.hasNode(JcrConstants.JCR_CONTENT))
            {
                resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
                return resourceNode.getProperty(JcrConstants.JCR_LASTMODIFIED).getDate().getTime();
            }
            
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get lastmodified for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public long getLength() throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            Node resourceNode = null;
            
            if (fileNode.hasNode(JcrConstants.JCR_CONTENT))
            {
                resourceNode = fileNode.getNode(JcrConstants.JCR_CONTENT);
                return resourceNode.getProperty(JcrConstants.JCR_DATA).getLength();
            }
            
            return 0;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get length for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public UserIdentity getCreator() throws AmetysRepositoryException
    {
        try
        {
            Node authorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATOR_NODE_NAME);
            return new UserIdentity(authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login").getString(), authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population").getString());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get creator for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    /**
     * Get the author from old revision
     * @param revision The revision
     * @return The user identity of the author or <code>null</code> if not found
     * @throws RepositoryException If an error occurred
     */
    public UserIdentity getAuthorFromRevision (String revision) throws RepositoryException
    {
        try
        {
            switchToRevision(revision);
            
            VersionHistory history = getVersionHistory();
            Node versionNode = history.getVersion(revision);
            Node frozenNode = versionNode.getNode(JcrConstants.JCR_FROZENNODE);
            
            if (frozenNode.hasNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATOR_NODE_NAME))
            {
                Node authorNode = frozenNode.getNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATOR_NODE_NAME);
                return new UserIdentity(authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login").getString(), authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population").getString());
            }
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get author from revision: " + revision + " of resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public String[] getKeywords() throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            if (!fileNode.hasProperty("ametys:keywords"))
            {
                return new String[0];
            }
            
            Value[] values = fileNode.getProperty("ametys:keywords").getValues();
            String[] result = new String[values.length];
            
            for (int i = 0; i < values.length; i++)
            {
                result[i] = values[i].getString();
            }
            
            return result;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get keywords for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    public String getKeywordsAsString() throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            if (!fileNode.hasProperty("ametys:keywords"))
            {
                return StringUtils.EMPTY;
            }
            
            StringBuilder sb = new StringBuilder();
            Value[] values = fileNode.getProperty("ametys:keywords").getValues();
            
            for (Value value : values)
            {
                if (sb.length() > 0)
                {
                    sb.append(", ");
                }
                
                sb.append(value.getString());
            }
            
            return sb.toString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get keywords for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    @Override
    protected void restoreFromNode(Node node) throws RepositoryException
    {
        super.restoreFromNode(node);
        
        // First remove node
        NodeIterator nit = getBaseNode().getNodes(JcrConstants.JCR_CONTENT);
        while (nit.hasNext())
        {
            nit.nextNode().remove();
        }
        
        NodeIterator new_nit = node.getNodes(JcrConstants.JCR_CONTENT);
        while (new_nit.hasNext())
        {
            copyNode(getBaseNode(), new_nit.nextNode());
        }
    }
    
    @Override
    public String getResourcePath() throws AmetysRepositoryException
    {
        return ((ExplorerNode) getParent()).getExplorerPath() + "/" + getName();
    }
    
    @Override
    public AmetysObject copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        try
        {
            String nodeTypeName = NodeTypeHelper.getNodeTypeName(getNode());
            
            JCRResource copiedResource = parent.createChild(name, nodeTypeName);
            
            copiedResource.setKeywords(getKeywords());
            copiedResource.setData(getInputStream(), getMimeType(), getLastModified(), getCreator());
            
            parent.saveChanges();
            
            return copiedResource;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error copying the collection " + getId() + " into " + parent.getId(), e);
        }
    }
    
    @Override
    public AmetysObject copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }
    
    // Dublin Core metadata. //
    
    @Override
    public String getDCTitle() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCTitle(this, getName());
    }
    
    @Override
    public void setDCTitle(String title) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCTitle(this, title);
    }
    
    @Override
    public String getDCCreator() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCCreator(this);
    }
    
    @Override
    public void setDCCreator(String creator) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCCreator(this, creator);
    }
    
    @Override
    public String[] getDCSubject() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCSubject(this, getKeywords());
    }
    
    @Override
    public void setDCSubject(String[] subject) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCSubject(this, subject);
    }
    
    @Override
    public String getDCDescription() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCDescription(this);
    }
    
    @Override
    public void setDCDescription(String description) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCDescription(this, description);
    }
    
    @Override
    public String getDCPublisher() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCPublisher(this);
    }
    
    @Override
    public void setDCPublisher(String publisher) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCPublisher(this, publisher);
    }
    
    @Override
    public String getDCContributor() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCContributor(this, UserIdentity.userIdentityToString(getCreator()));
    }
    
    @Override
    public void setDCContributor(String contributor) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCContributor(this, contributor);
    }
    
    @Override
    public Date getDCDate() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCDate(this, getLastModified());
    }
    
    @Override
    public void setDCDate(Date date) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCDate(this, date);
    }
    
    @Override
    public String getDCType() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCType(this, _getDefaultDCType());
    }
    
    private String _getDefaultDCType ()
    {
        String mimetype = getMimeType();
        
        if (mimetype == null)
        {
            return DCMITypes.TEXT;
        }
        else if (mimetype.startsWith("image"))
        {
            return DCMITypes.IMAGE;
        }
        else if (mimetype.startsWith("video") || "application/x-shockwave-flash".equals(mimetype))
        {
            return DCMITypes.INTERACTIVERESOURCE;
        }
        else if (mimetype.startsWith("audio"))
        {
            return DCMITypes.SOUND;
        }
        
        return DCMITypes.TEXT;
    }
    
    
    @Override
    public void setDCType(String type) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCType(this, type);
    }
    
    @Override
    public String getDCFormat() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCFormat(this, getMimeType());
    }
    
    @Override
    public void setDCFormat(String format) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCFormat(this, format);
    }
    
    @Override
    public String getDCIdentifier() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCIdentifier(this, getId());
    }
    
    @Override
    public void setDCIdentifier(String identifier) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCIdentifier(this, identifier);
    }
    
    @Override
    public String getDCSource() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCSource(this);
    }
    
    @Override
    public void setDCSource(String source) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCSource(this, source);
    }
    
    @Override
    public String getDCLanguage() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCLanguage(this);
    }
    
    @Override
    public void setDCLanguage(String language) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCLanguage(this, language);
    }
    
    @Override
    public String getDCRelation() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCRelation(this);
    }
    
    @Override
    public void setDCRelation(String relation) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCRelation(this, relation);
    }
    
    @Override
    public String getDCCoverage() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCCoverage(this, getDCLanguage());
    }
    
    @Override
    public void setDCCoverage(String coverage) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCCoverage(this, coverage);
    }
    
    @Override
    public String getDCRights() throws AmetysRepositoryException
    {
        return DublinCoreHelper.getDCRights(this);
    }
    
    @Override
    public void setDCRights(String rights) throws AmetysRepositoryException
    {
        DublinCoreHelper.setDCRights(this, rights);
    }

    public Date getCreationDate() throws AmetysRepositoryException
    {
        Node fileNode = getNode();
        try
        {
            if (!fileNode.hasProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATION_DATE))
            {
                return null;
            }
            
            return fileNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATION_DATE).getDate().getTime();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get creation date for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }

    public void setCreationDate(Date creationDate)
    {
        Node fileNode = getNode();
        try
        {
            Calendar calendar = new GregorianCalendar();
            calendar.setTime(creationDate);
            fileNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":" + CREATION_DATE, calendar);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set create date for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }

    public UserIdentity getLastContributor() throws AmetysRepositoryException
    {
        try
        {
            Node authorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CONTRIBUTOR_NODE_NAME);
            return new UserIdentity(authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login").getString(), authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population").getString());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot get last contributor for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }

    public void setLastContributor(UserIdentity lastContributor)
    {
        try
        {
            Node lastContributorNode = null;
            if (getNode().hasNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CONTRIBUTOR_NODE_NAME))
            {
                lastContributorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CONTRIBUTOR_NODE_NAME);
            }
            else
            {
                lastContributorNode = getNode().addNode(RepositoryConstants.NAMESPACE_PREFIX + ":" + CONTRIBUTOR_NODE_NAME, RepositoryConstants.USER_NODETYPE);
            }
            lastContributorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", lastContributor.getLogin());
            lastContributorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", lastContributor.getPopulationId());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Cannot set last contributor for resource " + this.getName() + " (" + this.getId() + ")", e);
        }
    }
    
    public void tag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.tag(this, tag);
    }

    public void untag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.untag(this, tag);
    }

    public Set<String> getTags() throws AmetysRepositoryException
    {
        return TaggableAmetysObjectHelper.getTags(this);
    }
    
    public TrashElement moveToTrash()
    {
        TrashElement trashElement = _getFactory()._getTrashElementDAO().createTrashElement(this, getName());
        
        // Clone the ametys object from the original session to trash session
        Node storedNode = NodeHelper.cloneNode(getNode(), trashElement.getNode());
        
        try
        {
            storedNode.setProperty("ametys-internal:path", this.getParentPath());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("failed to store the content path", e);
        }

        // Remove the node from the original session
        remove();
        
        return trashElement;
    }
    
    public JCRResource restoreFromTrash() throws UnknownParentException
    {
        try
        {
            Property property = getNode().getProperty("ametys-internal:path");
            String parentPath = property.getValue().getString();
            property.remove();
            JCRTraversableAmetysObject parent;
            try
            {
                parent = _getFactory()._getResolver().resolveByPath(parentPath);
            }
            catch (UnknownAmetysObjectException e)
            {
                throw new UnknownParentException("No parent available at path " + parentPath, e);
            }
            
            // Get the node name, can be adjust if another content has already the same node name
            String nodeName = NameHelper.getUniqueAmetysObjectName(parent, getName(), NameComputationMode.USER_FRIENDLY, true);
            // Clone the ametys object from the trash session to default session
            NodeHelper.cloneNode(getNode(), parent.getNode(), nodeName);
            
            // Remove the node from the trash session
            remove();
            
            return parent.getChild(nodeName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("failed to store the content path", e);
        }
    }
}
