/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.jcr;

import java.util.List;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CopiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.jcr.NameHelper.NameComputationMode;
import org.ametys.plugins.repository.jcr.NodeHelper;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.trash.TrashElement;
import org.ametys.plugins.repository.trash.TrashableAmetysObject;
import org.ametys.plugins.repository.trash.UnknownParentException;

/**
 * {@link ExplorerNode} representing a collection of resources.
 * @param <F> the actual type of factory.
 */
public class JCRResourcesCollection<F extends JCRResourcesCollectionFactory> extends DefaultTraversableAmetysObject<F> implements ModifiableResourceCollection, CopiableAmetysObject, ModifiableModelLessDataAwareAmetysObject, TrashableAmetysObject
{
    /** application id for resources collections. */
    public static final String APPLICATION_ID = "Ametys.plugins.explorer.applications.resources.Resources";
    
    /** Constants for description metadata */
    protected static final String DATA_DESCRIPTION = "description";
    
    /**
     * Constructor
     * @param node The jcr node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public JCRResourcesCollection(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }

    @Override
    public String getIconCls()
    {
        return "ametysicon-folder249";
    }
    
    @Override
    public String getApplicationId()
    {
        return APPLICATION_ID;
    }
    
    @Override
    public String getCollectionType()
    {
        return JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE;
    }
    
    @Override
    public String getResourceType()
    {
        return JCRResourceFactory.RESOURCES_NODETYPE;
    }
    
    @Override
    public String getResourcePath() throws AmetysRepositoryException
    {
        return getExplorerPath();
    }

    @Override
    public String getExplorerPath()
    {
        AmetysObject parent = getParent();
        
        if (parent instanceof ExplorerNode)
        {
            return ((ExplorerNode) parent).getExplorerPath() + "/" + getName();
        }
        else
        {
            return "";
        }
    }
    
    public boolean hasChildResources() throws AmetysRepositoryException
    {
        AmetysObjectIterable<AmetysObject> children = getChildren();
        for (AmetysObject child : children)
        {
            if (child instanceof Resource)
            {
                return true;
            }
        }

        return false;
    }
    
    public boolean hasChildExplorerNodes() throws AmetysRepositoryException
    {
        AmetysObjectIterable<AmetysObject> children = getChildren();
        for (AmetysObject child : children)
        {
            if (child instanceof ExplorerNode)
            {
                return true;
            }
        }

        return false;
    }
    
    @Override
    public AmetysObject copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        try
        {
            String nodeTypeName = NodeTypeHelper.getNodeTypeName(getNode());
            
            JCRResourcesCollection copiedCollection = parent.createChild(name, nodeTypeName);
            parent.saveChanges();
            
            for (AmetysObject object : getChildren())
            {
                if (object instanceof CopiableAmetysObject)
                {
                    ((CopiableAmetysObject) object).copyTo(copiedCollection, object.getName());
                }
            }
            
            return copiedCollection;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error copying the collection " + getId() + " into " + parent.getId(), e);
        }
    }
    
    @Override
    public AmetysObject copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }

    @Override
    public String getDescription()
    {
        return getValue(DATA_DESCRIPTION, null);
    }

    @Override
    public void setDescription(String description)
    {
        setValue(DATA_DESCRIPTION, description);
    }
    
    public ModifiableModelLessDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getDataTypesExtensionPoint(), repositoryData);
    }
    
    public TrashElement moveToTrash()
    {
        TrashElement trashElement = _getFactory()._getTrashElementDAO().createTrashElement(this, getName());
        // Clone the ametys object from the original session to trash session
        Node storedNode = NodeHelper.cloneNode(getNode(), trashElement.getNode());
        
        try
        {
            storedNode.setProperty("ametys-internal:path", this.getParentPath());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("failed to store the content path", e);
        }

        // Remove the node from the original session
        remove();
        
        return trashElement;
    }
    
    public JCRResourcesCollection restoreFromTrash() throws UnknownParentException
    {
        try
        {
            Property property = getNode().getProperty("ametys-internal:path");
            String parentPath = property.getValue().getString();
            property.remove();
            JCRTraversableAmetysObject parent;
            try
            {
                parent = _getFactory()._getResolver().resolveByPath(parentPath);
            }
            catch (UnknownAmetysObjectException e)
            {
                throw new UnknownParentException("No parent available at path " + parentPath, e);
            }
            
            // Get the node name, can be adjust if another content has already the same node name
            String nodeName = NameHelper.getUniqueAmetysObjectName(parent, getName(), NameComputationMode.USER_FRIENDLY, true);
            
            // Clone the ametys object from the trash session to default session
            NodeHelper.cloneNode(getNode(), parent.getNode(), nodeName);
            
            // Remove the node from the trash session
            remove();
            
            return parent.getChild(nodeName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("failed to store the content path", e);
        }
    }
}
