/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.metadata.populate;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.tika.metadata.DublinCore;
import org.apache.tika.metadata.Metadata;

import org.ametys.plugins.explorer.resources.ModifiableResource;

/**
 * {@link ResourceMetadataPopulator} which populates Dublin Core metadata.
 */
public class DublinCorePopulator extends AbstractLogEnabled implements ResourceMetadataPopulator, Configurable
{
    
    private Set<String> _types;
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _types = new HashSet<>();
        
        Configuration[] types = configuration.getChild("types").getChildren("type");
        for (Configuration type : types)
        {
            _types.add(type.getValue());
        }
    }
    
    @Override
    public Set<String> getTypes()
    {
        return Collections.unmodifiableSet(_types);
    }

    @Override
    public void populate(ModifiableResource resource, Metadata metadata)
    {
        String creator = metadata.get(DublinCore.CREATOR);
        String contributor = metadata.get(DublinCore.CONTRIBUTOR);
        String coverage = metadata.get(DublinCore.COVERAGE);
        Date date = metadata.getDate(DublinCore.DATE);
        String description = metadata.get(DublinCore.DESCRIPTION);
        String format = metadata.get(DublinCore.FORMAT);
        String identifier = metadata.get(DublinCore.IDENTIFIER);
        String language = metadata.get(DublinCore.LANGUAGE);
        String publisher = metadata.get(DublinCore.PUBLISHER);
        String relation = metadata.get(DublinCore.RELATION);
        String rights = metadata.get(DublinCore.RIGHTS);
        String source = metadata.get(DublinCore.SOURCE);
        String[] subject = metadata.getValues(DublinCore.SUBJECT);
        String title = metadata.get(DublinCore.TITLE);
        String type = metadata.get(DublinCore.TYPE);
        
        _setDCCreator(resource, creator);
        _setDCContributor(resource, contributor);
        _setDCCoverage(resource, coverage);
        _setDCDate(resource, date);
        _setDCDescription(resource, description);
        _setDCFormat(resource, format);
        _setDCIdentifier(resource, identifier);
        _setDCLanguage(resource, language);
        _setDCPublisher(resource, publisher);
        _setDCRelation(resource, relation);
        _setDCRights(resource, rights);
        _setDCSource(resource, source);
        _setDCSubject(resource, subject);
        _setDCTitle(resource, title);
        _setDCType(resource, type);
    }
    
    /**
     * Set DC Creator
     * @param resource The resource to edit
     * @param creator The creator
     */
    protected void _setDCCreator(ModifiableResource resource, String creator)
    {
        if (creator != null)
        {
            resource.setDCCreator(creator);
        }
    }
    
    /**
     * Set DC Contributor
     * @param resource The resource to edit
     * @param contributor The contributor login
     */
    protected void _setDCContributor(ModifiableResource resource, String contributor)
    {
        if (contributor != null)
        {
            resource.setDCContributor(contributor);
        }
    }
    
    /**
     * Set DC Coverage
     * @param resource The resource to edit
     * @param coverage The coverage
     */
    protected void _setDCCoverage(ModifiableResource resource, String coverage)
    {
        if (coverage != null)
        {
            resource.setDCCoverage(coverage);
        }
    }
    
    /**
     * Set DC Date
     * @param resource The resource to edit
     * @param date The date
     */
    protected void _setDCDate(ModifiableResource resource, Date date)
    {
        if (date != null)
        {
            resource.setDCDate(date);
        }
    }
    
    /**
     * Set DC Description
     * @param resource The resource to edit
     * @param description The description
     */
    protected void _setDCDescription(ModifiableResource resource, String description)
    {
        if (description != null)
        {
            resource.setDCDescription(description);
        }
    }
    
    /**
     * Set DC Format
     * @param resource The resource to edit
     * @param format The format
     */
    protected void _setDCFormat(ModifiableResource resource, String format)
    {
        if (format != null)
        {
            resource.setDCFormat(format);
        }
    }
    
    /**
     * Set DC Identifier
     * @param resource The resource to edit
     * @param identifier The identifier
     */
    protected void _setDCIdentifier(ModifiableResource resource, String identifier)
    {
        if (identifier != null)
        {
            resource.setDCIdentifier(identifier);
        }
    }
    
    /**
     * Set DC language
     * @param resource The resource to edit
     * @param language The language
     */
    protected void _setDCLanguage(ModifiableResource resource, String language)
    {
        if (language != null)
        {
            resource.setDCLanguage(language);
        }
    }
    
    /**
     * Set DC Publisher
     * @param resource The resource to edit
     * @param publisher The publisher
     */
    protected void _setDCPublisher(ModifiableResource resource, String publisher)
    {
        if (publisher != null)
        {
            resource.setDCPublisher(publisher);
        }
    }
    
    /**
     * Set DC Relation
     * @param resource The resource to edit
     * @param relation The relation
     */
    protected void _setDCRelation(ModifiableResource resource, String relation)
    {
        if (relation != null)
        {
            resource.setDCRelation(relation);
        }
    }
    
    /**
     * Set DC rights
     * @param resource The resource to edit
     * @param rights The rights
     */
    protected void _setDCRights(ModifiableResource resource, String rights)
    {
        if (rights != null)
        {
            resource.setDCRights(rights);
        }
    }
    
    /**
     * Set DC Source
     * @param resource The resource to edit
     * @param source The source
     */
    protected void _setDCSource(ModifiableResource resource, String source)
    {
        if (source != null)
        {
            resource.setDCSource(source);
        }
    }
    
    /**
     * Set DC Subject
     * @param resource The resource to edit
     * @param subject The subjects
     */
    protected void _setDCSubject(ModifiableResource resource, String[] subject)
    {
        if (subject != null && subject.length > 0)
        {
            resource.setDCSubject(subject);
        }
    }
    
    /**
     * Set DC Title
     * @param resource The resource to edit
     * @param title The title
     */
    protected void _setDCTitle(ModifiableResource resource, String title)
    {
        if (title != null)
        {
            resource.setDCTitle(title);
        }
    }
    
    /**
     * Set DC Type
     * @param resource The resource to edit
     * @param type The type
     */
    protected void _setDCType(ModifiableResource resource, String type)
    {
        if (type != null)
        {
            resource.setDCType(type);
        }
    }
}
