/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.metadata.populate;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * Extension point for {@link ResourceMetadataPopulator}s.
 */
public class ResourceMetadataPopulatorExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<ResourceMetadataPopulator>
{
    /** Avalon Role */
    public static final String ROLE = ResourceMetadataPopulatorExtensionPoint.class.getName();

    private Map<String, Set<ResourceMetadataPopulator>> _types;
    
    @Override
    public void initializeExtensions() throws Exception
    {
        super.initializeExtensions();
        
        initializePopulatorMap();
    }
    
    /**
     * Initialize the populator map by type.
     */
    protected void initializePopulatorMap()
    {
        _types = new HashMap<>();
        
        for (String id : getExtensionsIds())
        {
            ResourceMetadataPopulator populator = getExtension(id);
            
            if (populator.getTypes() == null || populator.getTypes().isEmpty())
            {
                // Default: use this populator for all the file types.
                addPopulator("", populator);
            }
            else
            {
                for (String type : populator.getTypes())
                {
                    addPopulator(type, populator);
                }
            }
        }
    }

    /**
     * Add the populator to the map.
     * @param type the mime type or the empty string.
     * @param populator the populator.
     */
    protected void addPopulator(String type, ResourceMetadataPopulator populator)
    {
        Set<ResourceMetadataPopulator> types;
        if (_types.containsKey(type))
        {
            types = _types.get(type);
        }
        else
        {
            types = new HashSet<>();
            _types.put(type, types);
        }
        
        types.add(populator);
    }
    
    /**
     * Returns the {@link ResourceMetadataPopulator}s corresponding to the given type.
     * @param type the MIME type.
     * @return a Collection of {@link ResourceMetadataPopulator}s, empty if there is no populator for the given type.
     */
    public Collection<ResourceMetadataPopulator> getPopulators(String type)
    {
        Set<ResourceMetadataPopulator> populators = new HashSet<>();
        
        // Add default populators.
        if (_types.containsKey(""))
        {
            populators.addAll(_types.get(""));
        }
        
        if (_types.containsKey(type))
        {
            populators.addAll(_types.get(type));
        }
        
        return populators;
    }
    
}
