/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.resources.readers;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.ResourceNotFoundException;
import org.apache.cocoon.caching.CacheableProcessingComponent;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceValidity;
import org.apache.excalibur.source.impl.validity.TimeStampValidity;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.cocoon.AbstractResourceReader;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;

/**
 * Reader for {@link Resource}
 */
public class AmetysResourceReader extends AbstractResourceReader implements Serviceable, CacheableProcessingComponent
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The resource */
    protected Resource _object;
    /** The right manager */
    protected RightManager _rightManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) sManager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) sManager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) sManager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    protected void doSetup(SourceResolver res, Map objModel, String src, Parameters par) throws ProcessingException, IOException
    {
        String id = par.getParameter("id", null);
        String path = par.getParameter("path", null);
        String version = par.getParameter("version", null);
        
        try
        {
            if (id != null)
            {
                _object = _resolver.resolveById(id);
            }
            else
            {
                _object = _resolver.resolveByPath(FilenameUtils.decode(path.substring(1)));
            }

            // Check user access
            checkUserAccess();
            
            if (!StringUtils.isEmpty(version) && _object instanceof VersionableAmetysObject)
            {
                ((VersionableAmetysObject) _object).switchToRevision(version);
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            if (id != null)
            {
                throw new ResourceNotFoundException(String.format("The resource with id '%s' does not exist", id));
            }
            else
            {
                throw new ResourceNotFoundException(String.format("The resource at path '%s' does not exist", path));
            }
        }
    }
    
    @Override
    protected InputStream getInputStream()
    {
        return _object.getInputStream();
    }
    
    @Override
    protected String getFilename()
    {
        return _object.getName();
    }
    
    @Override
    protected String getEncodedFilename()
    {
        String name = _object.getName();
        return FilenameUtils.encodeName(name);
    }
    
    @Override
    protected long getLength()
    {
        return _object.getLength();
    }
    
    /**
     * Check the user access
     * @throws AuthorizationRequiredException if authorization is required
     * @throws AccessDeniedException if user has no access
     */
    protected void checkUserAccess() throws AuthorizationRequiredException, AccessDeniedException
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        if (user == null)
        {
            // Check anonymous access
            if (!_rightManager.hasAnonymousReadAccess(_object.getParent()))
            {
                throw new AuthorizationRequiredException(null);
            }
        }
        else if (!_rightManager.hasReadAccess(user, _object.getParent()))
        {
            throw new AccessDeniedException("User " + user + " has no right to access the resource " + _object.getId());
        }
    }
    
    @Override
    public Serializable getKey()
    {
        return _object.getId() + getKeySuffix();
    }

    @Override
    public SourceValidity getValidity()
    {
        return new TimeStampValidity(getLastModified());
    }
    
    @Override
    public long getLastModified()
    {
        return _object.getLastModified().getTime();
    }
    
    @Override
    public String getMimeType()
    {
        return _object.getMimeType();
    }
    
    @Override
    public void recycle()
    {
        super.recycle();
        _object = null;
    }
}
