/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.rights;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link AccessController} for a {@link Resource}
 */
public class ResourceAccessController extends AbstractHierarchicalAccessController<AmetysObject>
{
    /** the resource context category */
    public static final I18nizableText RESOURCE_CONTEXT_CATEGORY = new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RIGHT_ASSIGNMENT_CONTEXT_RESOURCES_LABEL");
    /** The explorer resources dao */
    protected ExplorerResourcesDAO _resourcesDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        
        _resourcesDAO = (ExplorerResourcesDAO) manager.lookup(ExplorerResourcesDAO.ROLE);
    }
    
    public boolean supports(Object object)
    {
        if (object instanceof ExplorerNode node)
        {
            return _isPartOfResourceExplorer(node.getPath());
        }
        else if (object instanceof Resource resource)
        {
            return _isPartOfResourceExplorer(resource.getPath());
        }
        return false;
    }

    private boolean _isPartOfResourceExplorer(String path)
    {
        for (Pattern pattern : _resourcesDAO.getExplorerNodePathPatterns())
        {
            if (pattern.matcher(path).matches())
            {
                return true;
            }
        }
        return false;
    }
    
    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object)
    {
        AmetysObject parent = object.getParent();
        if (parent instanceof ExplorerNode)
        {
            return Collections.singleton(parent);
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected Set<? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            Set<Object> resourcesRootContexts = new HashSet<>();
            resourcesRootContexts.addAll(_resourcesDAO.getResourcesRootNodes());
            return resourcesRootContexts;
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        Map<String, I18nizableTextParameter> params = Map.of("name", getObjectLabel(object));
        if (object instanceof Resource)
        {
            return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RESOURCE_ACCESS_CONTROLLER_RESOURCE_CONTEXT_LABEL", params);
            
        }
        else if (object instanceof ExplorerNode)
        {
            return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RESOURCE_ACCESS_CONTROLLER_EXPLORER_NODE_CONTEXT_LABEL", params);
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object) throws RightsException
    {
        if (object instanceof Resource resource)
        {
            return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RESOURCE_ACCESS_CONTROLLER_PATH",
                    Map.of("parent", getObjectLabel(resource.getParent()),
                           "child", new I18nizableText(resource.getName()))
                    );
        }
        else if (object instanceof ExplorerNode node)
        {
            if (_resourcesDAO.getResourcesRootNodes().contains(node))
            {
                return _resourcesDAO.getRootNodeLabel(node);
            }
            else
            {
                return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RESOURCE_ACCESS_CONTROLLER_PATH",
                        Map.of("parent", getObjectLabel(node.getParent()),
                               "child", new I18nizableText(node.getName()))
                        );
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        return RESOURCE_CONTEXT_CATEGORY;
    }
}
