/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.explorer.statistics;

import java.util.HashMap;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.statistics.Statistics;
import org.ametys.runtime.plugins.admin.statistics.StatisticsNode;
import org.ametys.runtime.plugins.admin.statistics.StatisticsProvider;
import org.ametys.runtime.plugins.admin.statistics.StatisticsValue;

/**
 * Explorer statistics
 */
public class ExplorerStatisticsProvider implements StatisticsProvider, Serviceable, PluginAware
{
    private static String __FOLDER_COUNT = "FOLDERCOUNT";
    private static String __RESOURCE_COUNT = "RESOURCECOUNT";
    private static String __TOTAL_SIZE = "TOTALSIZE";
    
    private String _id;
    private ExplorerResourcesDAO _explorerResourcesDAO;

    public void service(ServiceManager manager) throws ServiceException
    {
        _explorerResourcesDAO = (ExplorerResourcesDAO) manager.lookup(ExplorerResourcesDAO.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public Statistics getStatistics()
    {
        HashMap<String, Long> values = new HashMap<>();
        values.put(__FOLDER_COUNT, 0L);
        values.put(__RESOURCE_COUNT, 0L);
        values.put(__TOTAL_SIZE, 0L);
        
        for (ExplorerNode explorerNode : _explorerResourcesDAO.getResourcesRootNodes())
        {
            _processResources((ResourceCollection) explorerNode, values);
        }
        
        return new StatisticsNode(
            _id,
            new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_STATISTICS_LABEL"),
            "ametysicon-folder249",
            values.get(__TOTAL_SIZE),
            List.of(
                new StatisticsValue(
                    "folders",
                    new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_STATISTICS_FOLDERS_LABEL"),
                    "ametysicon-folder250",
                    values.get(__FOLDER_COUNT)
                ),
                new StatisticsValue(
                    "files",
                    new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_STATISTICS_FILES_LABEL"),
                    "ametysicon-document77",
                    values.get(__RESOURCE_COUNT)
                )                
            ),
            true
        );
    }
    
    private void _processResources(TraversableAmetysObject resourceContainer, HashMap<String, Long> values)
    {
        AmetysObjectIterable<? extends AmetysObject> objects = resourceContainer.getChildren();
        
        for (AmetysObject object : objects)
        {
            // Traverse the child nodes if depth < 0 (generate all) or depth > 0 (we're not in the last level).
            if (object instanceof ResourceCollection)
            {
                values.put(__FOLDER_COUNT, values.get(__FOLDER_COUNT) + 1);
                
                _processResources((ResourceCollection) object, values);
            }
            else if (object instanceof Resource)
            {
                Resource resource = (Resource) object;
                
                values.put(__RESOURCE_COUNT, values.get(__RESOURCE_COUNT) + 1);
                values.put(__TOTAL_SIZE, values.get(__TOTAL_SIZE) + resource.getLength());
            }
        }
    }
}
