/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button which state depends on whether the node can be modified or not.
 * A node can be modified only if it's not locked and not checked-in.
 */
Ext.define('Ametys.repository.controller.SmartNodeController',
{
    extend: 'Ametys.repository.controller.RepositoryButtonController',
    
    /**
     * @cfg {String} [enable-on-unlock-only] Set to 'true' to enable button only when the node is unlocked.
     */
    /**
     * @private
     * @property {Boolean} [_enableOnUnlockOnly=false] See #cfg-enable-on-unlock-only.
     */
    
    /**
     * @cfg {String} [enable-on-checkout-only] Set to 'true' to enable button only when the node is checked-out.
     */
    /**
     * @private
     * @property {Boolean} [_enableOnCheckoutOnly=false] See #cfg-enable-on-unlock-only.
     */
    
    /**
     * @cfg {String} [enable-on-checkin-only] Set to 'true' to enable button only when the node is not checked-out.
     */
    /**
     * @private
     * @property {Boolean} [_enableOnCheckinOnly=false] See #cfg-enable-on-unlock-only.
     */
    
    /**
     * @cfg {String} [icon-glyph-checkout] The specific glyph icon to use when the node is checked-out.
     */
    /**
     * @private
     * @property {Boolean} _iconGlyphCheckout See #cfg-icon-glyph-checkout.
     */
    /**
     * @cfg {String} [icon-decorator-checkout] The specific decorator to use when the node is checked-out.
     */
    /**
     * @private
     * @property {Boolean} _iconDecoratorCheckout See #cfg-icon-decorator-checkout.
     */
    
    /**
     * @cfg {String} [icon-glyph-checkin] The specific glyph icon to use when the node is not checked-out.
     */
    /**
     * @private
     * @property {Boolean} _iconGlyphCheckin See #cfg-icon-glyph-checkin.
     */
    /**
     * @cfg {String} [icon-decorator-checkin] The specific decorator to use when the node is not checked-out.
     */
    /**
     * @private
     * @property {Boolean} _iconDecoratorCheckin See #cfg-icon-decorator-checkin.
     */
    
    /**
     * @cfg {String} [icon-glyph-locked] The specific glyph icon to use when the node is locked.
     */
    /**
     * @private
     * @property {Boolean} _iconGlyphLocked See #cfg-icon-glyph-locked.
     */
    /**
     * @cfg {String} [icon-decorator-locked] The specific decorator to use when the node is locked.
     */
    /**
     * @private
     * @property {Boolean} _iconDecoratorLocked See #cfg-icon-decorator-locked.
     */
    
    /**
     * @cfg {String} [icon-glyph-unlocked] The specific glyph icon to use when the node is unlocked.
     */
    /**
     * @private
     * @property {Boolean} _iconGlyphUnlocked See #cfg-icon-glyph-unlocked.
     */
    /**
     * @cfg {String} [icon-decorator-unlocked] The specific decorator to use when the node is unlocked.
     */
    /**
     * @private
     * @property {Boolean} _iconDecoratorUnlocked See #cfg-icon-decorator-unlocked.
     */
    
    constructor: function (config)
    {
        this.callParent(arguments);
        
        this._enableOnUnlockOnly = config['enable-on-unlock-only'] == "true";
        this._enableOnLockOnly = config['enable-on-lock-only'] == "true";
        this._enableOnCheckoutOnly = config['enable-on-checkout-only'] == "true";
        this._enableOnCheckinOnly = config['enable-on-checkin-only'] == "true";
        
        // Icons
        this._iconGlyphCheckout = this.getInitialConfig('icon-glyph-checkout') || this.getInitialConfig('icon-glyph');
        this._iconDecoratorCheckout = this.getInitialConfig('icon-decorator-checkout') || this.getInitialConfig('icon-decorator');
        
        this._iconGlyphCheckin = this.getInitialConfig('icon-glyph-checkedin') || this.getInitialConfig('icon-glyph');
        this._iconDecoratorCheckin = this.getInitialConfig('icon-decorator-checkedin') || this.getInitialConfig('icon-decorator');
        
        this._iconGlyphLocked = this.getInitialConfig('icon-glyph-locked') || this.getInitialConfig('icon-glyph');
        this._iconDecoratorLocked = this.getInitialConfig('icon-decorator-locked') || this.getInitialConfig('icon-decorator');
        
        this._iconGlyphUnlocked = this.getInitialConfig('icon-glyph-unlocked') || this.getInitialConfig('icon-glyph');
        this._iconDecoratorUnlocked = this.getInitialConfig('icon-decorator-unlocked') || this.getInitialConfig('icon-decorator');
        
        Ametys.message.MessageBus.on(Ametys.message.Message.SELECTION_CHANGED, this.refreshIfMatchingMessage, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this.refreshIfMatchingMessage, this);
    },
    
    updateState: function()
    {
        var targets = this.getMatchingTargets();
        if (targets.length > 0)
        {
	        this._updateState(targets[0]);
        }
    },
    
    /**
     * Updates the button state.
     * @param {Ametys.message.MessageTarget} target the selected target.
     * @private
     */
    _updateState: function(target)
    {
        // Reset state
        this.disable();
        
        var ok = true;
        if (this._enableOnUnlockOnly)
        {
            if (target.getParameters().locked)
            {
                ok = false;
	            this.setAdditionalDescription(this.getInitialConfig('description-locked') || '');
	            this.setGlyphIcon (this._iconGlyphLocked);
	            this.setIconDecorator (this._iconDecoratorLocked);
            }
            else
            {
                this.setAdditionalDescription(this.getInitialConfig('description-unlocked') || '');
                this.setGlyphIcon (this._iconGlyphUnlocked);
                this.setIconDecorator (this._iconDecoratorUnlocked);
            }
        }
        
        if (this._enableOnLockOnly)
        {
            if (target.getParameters().locked)
            {
                this.setAdditionalDescription(this.getInitialConfig('description-locked') || '');
                this.setGlyphIcon (this._iconGlyphLocked);
                this.setIconDecorator (this._iconDecoratorLocked);
            }
            else if (target.getParameters().lockable === false)
            {
                ok = false;
                this.setAdditionalDescription(this.getInitialConfig('description-not-lockable') || this.getInitialConfig('description-unlocked') || '');
                this.setGlyphIcon (this._iconGlyphUnlocked);
                this.setIconDecorator (this._iconDecoratorUnlocked);
            }
            else
            {
                ok = false;
                this.setAdditionalDescription(this.getInitialConfig('description-unlocked') || '');
                this.setGlyphIcon (this._iconGlyphUnlocked);
                this.setIconDecorator (this._iconDecoratorUnlocked);
            }
        }
        
        
        if (this._enableOnCheckoutOnly)
        {
            if (target.getParameters().checkedOut)
            {
                this.setAdditionalDescription(this.getInitialConfig('description-checkout') || '');
	            this.setGlyphIcon (this._iconGlyphCheckout);
	            this.setIconDecorator (this._iconDecoratorCheckout);
            }
            else
            {
                ok = false;
                this.setAdditionalDescription(this.getInitialConfig('description-checkin') || '');
	            this.setGlyphIcon (this._iconGlyphCheckin);
	            this.setIconDecorator (this._iconDecoratorCheckin);
            }
        }
        
        if (this._enableOnCheckinOnly)
        {
            if (target.getParameters().checkedOut)
            {
                ok = false;
                this.setAdditionalDescription(this.getInitialConfig('description-checkout') || '');
                this.setGlyphIcon (this._iconGlyphCheckout);
                this.setIconDecorator (this._iconDecoratorCheckout);
            }
            else
            {
                this.setAdditionalDescription(this.getInitialConfig('description-checkin') || '');
                this.setGlyphIcon (this._iconGlyphCheckin);
                this.setIconDecorator (this._iconDecoratorCheckin);
            }
        }
        
        if (ok)
        {
            this.enable();
        }
        
        if (!this._enableOnUnlockOnly && !this._enableOnLockOnly && !this._enableOnCheckinOnly && !this._enableOnCheckoutOnly)
        {
            this.setAdditionalDescription('');
	        this.setGlyphIcon (this._iconGlyph);
	        this.setIconDecorator (this._iconDecorator);
        }
    }

});
