/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Model for repository descriptor.
 * @private
 */
Ext.define('Ametys.repository.tool.RepositoryDescriptorsTool.RepoDescriptor', {
    extend: 'Ext.data.Model',
    fields: [
        {name: 'key', mapping: '@key'},
        {name: 'standard', mapping: '@standard', type: 'boolean'},
        {name: 'singleValue', mapping: '@singleValue', type: 'boolean'}
    ],
    hasMany: {model: 'Ametys.repository.tool.RepositoryDescriptorsTool.RepoDescriptorValue', name: 'values'}
});

/**
 * Model for repository descriptor value.
 * @private
 */
Ext.define("Ametys.repository.tool.RepositoryDescriptorsTool.RepoDescriptorValue", {
    extend: 'Ext.data.Model',
    fields: [
        {name: 'value', mapping: '@value'}
    ],
    proxy: {
        type: 'memory',
        reader: {
            type: 'xml',
            rootProperty: 'values',
            record: 'value'
        }
    }
});

/**
 * Tool which shows the repository descriptors (properties).
 * @private
 */
Ext.define('Ametys.repository.tool.RepositoryDescriptorsTool',
{
    extend: 'Ametys.repository.tool.RepositoryTool',
    
    /**
     * @property {Ext.grid.Panel} grid The grid panel.
     * @private
     */
    grid: null,
    
    createPanel: function()
    {
        var store = Ext.create('Ext.data.Store', {
            model: 'Ametys.repository.tool.RepositoryDescriptorsTool.RepoDescriptor',
            proxy: {
                type: 'ametys',
                workspace: 'repository',
                url: 'repository/descriptors',
                reader: {
                    type: 'xml',
                    rootProperty: 'repository',
                    record: 'descriptor'
                }
            }
        });
        
        this.grid = Ext.create('Ext.grid.Panel', {
            border: false,
            scrollable: true,
            disableSelection: true,
            
            stateful: true,
            stateId: this.self.getName() + "$grid",
            store: store,
            
            columns: [
                {text: "{{i18n PLUGINS_REPOSITORYAPP_DESCRIPTORS_COLUMN_KEY}}", flex: 0.4, dataIndex: 'key', renderer: this._renderLabel},
                {text: "{{i18n PLUGINS_REPOSITORYAPP_DESCRIPTORS_COLUMN_VALUE}}", flex: 0.6, renderer: this._renderValue}
            ]
        });
        
        return this.grid;
    },
    
    setParams: function ()
    {
        this.callParent(arguments);
        this.grid.getStore().load();
    },

    getMBSelectionInteraction: function() 
    {
        return Ametys.tool.Tool.MB_TYPE_NOSELECTION;
    },
    
    getType: function()
    {
        return Ametys.tool.Tool.TYPE_REPOSITORY;  
    },
    
    /**
     * @private
     * Render function for descriptors' label
     * @param {String} label the label
     * @return {String} the formatted label
     */
    _renderLabel: function (label)
    {
        return '<b>' + label + '</b>';
    },
    
    /**
     * Render the value(s) of a descriptor as a String.
     * @param {Object} value The data value.
     * @param {Object} metaData A collection of metadata about the current cell.
     * @param {Ext.data.Model} record The record for the current row.
     * @private
     */
    _renderValue: function(value, metaData, record)
    {
        var valuesStr = '';
        
        record.values().each(function(val, index, length) {
            if (index > 0)
            {
                valuesStr += ', ';
            }
            valuesStr += val.get('value');
        });
        
        return valuesStr;
    }
    
});

