/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Class providing helper methods to format values.
 * @private
 */
Ext.define('Ametys.repository.utils.Utils',
{
    singleton: true,
    
    /**
     * Get a string representation of a String repository property. 
     * @param {String[]} values The property values (single-valued array if the property is not multiple).
     * @param {Boolean} multiple true if the property is multi-valued, false otherwise.
     * @param {Boolean} [escape=false] true to escape html tags
     */
    formatStringValue: function(values, multiple, escape)
    {
        return this.getMultipleValueDisplay(values, '<br/>', null, escape);
    },
    
    /**
     * Get a string representation of a Date repository property.
     * @param {String[]} values The property values (single-valued array if the property is not multiple).
     * @param {Boolean} multiple true if the property is multi-valued, false otherwise.
     */
    formatDateValue: function(values, multiple)
    {
        var date = Ext.Date.parse(values[0], Ext.Date.patterns.ISO8601DateTime);
        return this.formatDateObjectValue(date, multiple);
    },
    
    /**
     * Get a string representation of a Date repository property.
     * @param {Date[]} date The property values (single-valued array if the property is not multiple).
     * @param {Boolean} multiple true if the property is multi-valued, false otherwise.
     */
    formatDateObjectValue: function(date, multiple)
    {
        return Ext.Date.format(date, "{{i18n plugin.repositoryapp:PLUGINS_REPOSITORYAPP_PROPERTY_DISPLAY_DATE_TIME}}");
    },
    
    /**
     * Get a string representation of a Binary repository property.
     * @param {Array} values The property values (single-valued array if the property is not multiple).
     * @param {Boolean} multiple true if the property is multi-valued, false otherwise.
     */
    formatBinaryValue: function(values, multiple)
    {
        return "<a href='#'>{{i18n plugin.repositoryapp:PLUGINS_REPOSITORYAPP_PROPERTY_DOWNLOAD_BUTTON_TEXT}}</a>";
    },
    
    /**
     * Get a string representation of a Reference repository property.
     * @param {Array} values The property values (single-valued array if the property is not multiple).
     * @param {Boolean} multiple true if the property is multi-valued, false otherwise.
     */
    formatReferenceValue: function(values, multiple)
    {
        return values != null && values.length != 0 ? '<a href="#">' + values[0] + '</a>' : '';
    },
    
    /**
     * Join an array of string values with a specific separator, trimming them if needed. 
     * @param {String[]} values The array of strings to concatenate.
     * @param {String} [separator=','] The separator to inject between strings.
     * @param {Boolean} [trim] true to trim each string value before concatenating.
     * @param {Boolean} [escape=false] true to escape html tags
     */
    getMultipleValueDisplay: function(values, separator, trim, escape)
    {
        var value = '',
            sep = separator || ',';
        
        for (var i = 0; i < values.length; i++)
        {
            if (i > 0)
            {
                value += sep;
            }
            
            var v = trim === true ? values[i].trim() : values[i];
            if (escape)
            {
                v = v.replace(/&/g, "&amp;").replace(/</g, "&lt;");
            }
            value += v;
        }
        
        return value;
    }
    
});
