/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repositoryapp;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.Credentials;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Context;
import org.apache.jackrabbit.api.JackrabbitRepository;

/**
 * Repository helper.
 */
public class RepositoryProvider extends AbstractLogEnabled implements Contextualizable, Component
{
    /** The avalon role. */
    public static final String ROLE = RepositoryProvider.class.getName();
    
    /** The session map key. */
    public static final String SESSION_JCR_SESSIONS_KEY = "org.ametys.repository.Sessions";
    
    /** The context repository key. */
    public static final String CONTEXT_REPOSITORY_KEY = "org.ametys.repository.Repository";
    
    /** The JNDI variable key. */
    public static final String CONTEXT_IS_JNDI_KEY = "org.ametys.repository.JNDI";
    
    /** The credentials object key. */
    private static final String _CONTEXT_CREDENTIALS_KEY = "org.ametys.repository.Credentials";
    
    /** The avalon context, initialized during the contextualize method */
    private org.apache.avalon.framework.context.Context _avalonContext;
    
    private Context _context;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _avalonContext = context;
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Get the repository.
     * @return the repository.
     */
    public Repository getRepository()
    {
        return (Repository) _context.getAttribute(CONTEXT_REPOSITORY_KEY);
    }
    
    /**
     * Get the credentials.
     * @return the credentials.
     */
    public Credentials getCredentials()
    {
        return (Credentials) _context.getAttribute(_CONTEXT_CREDENTIALS_KEY);
    }
    
    /**
     * Determines if the repository is obtained through JNDI
     * @return true if the repository is obtained through JNDI
     */
    public boolean isJndi()
    {
        return (Boolean) _context.getAttribute(CONTEXT_IS_JNDI_KEY);
    }
    
    /**
     * Register a repository.
     * @param repository The repository to register
     * @param credentials The credentials to connect
     * @param jndi True to connect using jndi
     */
    public void registerRepository(Repository repository, Credentials credentials, boolean jndi)
    {
        _context.setAttribute(CONTEXT_REPOSITORY_KEY, repository);
        _context.setAttribute(_CONTEXT_CREDENTIALS_KEY, credentials);
        _context.setAttribute(CONTEXT_IS_JNDI_KEY, jndi);
    }
    
    /**
     * Get the JCR session map.
     * @return a Map of the JCR sessions, indexed by workspace name.
     */
    @SuppressWarnings("unchecked")
    public Map<String, Session> getSessions()
    {
        org.apache.cocoon.environment.Session httpSession = ContextHelper.getRequest(_avalonContext).getSession();
        
        Map<String, Session> sessions = (Map<String, Session>) httpSession.getAttribute(SESSION_JCR_SESSIONS_KEY);
        if (sessions == null)
        {
            sessions = new HashMap<>();
            httpSession.setAttribute(SESSION_JCR_SESSIONS_KEY, sessions);
        }
        
        return sessions;
    }
    
    /**
     * Get a session on the given workspace.
     * @param workspaceName the workspace name.
     * @return the JCR session.
     * @throws RepositoryException if an error occurred
     */
    public Session getSession(String workspaceName) throws RepositoryException
    {
        Map<String, Session> sessions = getSessions();
        
        Session session = sessions.get(workspaceName);
        
        if (session == null || !session.isLive())
        {
            Repository repository = getRepository();
            Credentials credentials = getCredentials();
            
            session = repository.login(credentials, workspaceName);
            sessions.put(workspaceName, session);
        }
        
        return session;
    }
    
    /**
     * Disconnect from the repository.
     */
    public void disconnect()
    {
        // Logout all the sessions.
        Map<String, Session> sessions = getSessions();
        for (Session session : sessions.values())
        {
            session.logout();
        }
        
        org.apache.cocoon.environment.Session httpSession = ContextHelper.getRequest(_avalonContext).getSession();
        httpSession.removeAttribute(SESSION_JCR_SESSIONS_KEY);
        
        // We should test if the component manager knows the repository (embedded mode),
        // and don't call shutdown in that case, but disconnect is called by LogoutAction, which is
        // only present in standalone mode, so we should be fine.
        Repository repository = getRepository();
        if (!isJndi() && repository instanceof JackrabbitRepository)
        {
            ((JackrabbitRepository) repository).shutdown();
        }
        
        // Destroy the connection information.
        _context.removeAttribute(CONTEXT_REPOSITORY_KEY);
        _context.removeAttribute(_CONTEXT_CREDENTIALS_KEY);
        _context.removeAttribute(CONTEXT_IS_JNDI_KEY);
    }
}
