/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repositoryapp.jcr;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.servlet.multipart.PartOnDisk;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repositoryapp.RepositoryProvider;
import org.ametys.workspaces.repository.jcr.RepositoryDao;

/**
 * Set value for a property (if the property does not exist, create the property)
 */
public class SetPropertyAction extends ServiceableAction
{
    
    /** The input date format. */
    protected static final DateFormat _DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
    
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _repositoryProvider = (RepositoryProvider) serviceManager.lookup(RepositoryProvider.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver sourceResolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String workspaceName = request.getParameter("workspace");
        
        Session session = _repositoryProvider.getSession(workspaceName);
        
        String nodePath = request.getParameter("path");
        String propertyName = request.getParameter("name");
        String[] values = request.getParameterValues("value");
        String type = request.getParameter("type");
        boolean multiple = Boolean.valueOf(request.getParameter("multiple"));
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to add property: '" + propertyName + "' to the node at path: '" + nodePath + "'");
        }
        
        String relPath = RepositoryDao.removeLeadingSlash(nodePath);
        
        // Get the parent node
        Node node = null;
        if (StringUtils.isEmpty(relPath))
        {
            node = session.getRootNode();
        }
        else
        {
            node = session.getRootNode().getNode(relPath);
        }
        
        // Get the type in jcr format (e.g. javax.jcr.PropertyType.BOOLEAN)
        int jcrType = javax.jcr.PropertyType.valueFromName(StringUtils.capitalize(type));
        
        try
        {
            // Add or update property to the node
            if (type.equalsIgnoreCase("Date"))
            {
                _setDate(node, propertyName, values);
            }
            else if (type.equalsIgnoreCase("Binary"))
            {
                _setBinary(session, request, node, propertyName);
            }
            else if (type.equalsIgnoreCase("Reference"))
            {
                _setReference(session, node, propertyName, values, multiple);
            }
            else
            {
                _setValue(node, propertyName, values, multiple, jcrType);
            }
            
            result.put("success", true);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to set value " + Arrays.toString(values) + " for property '" + propertyName + "'", e);
            result.put("success", false);
            result.put("message", e.toString());
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    private void _setDate(Node node, String propertyName, String[] value) throws RepositoryException
    {
        //create calendar date from string value
        ZonedDateTime zdt = DateUtils.parseZonedDateTime(value[0]);
        Calendar cal = GregorianCalendar.from(zdt);
        
        node.setProperty(propertyName, cal);
    }
    
    private void _setBinary(Session session, Request request, Node node, String propertyName) throws RepositoryException, IOException
    {
        // The upload was either sent as the property name (when modified) or as "fieldValue" (when added).
        PartOnDisk uploadedFilePart = (PartOnDisk) request.get(propertyName);
        if (uploadedFilePart == null)
        {
            uploadedFilePart = (PartOnDisk) request.get("fieldValue");
        }
        
        File uploadedFile = (uploadedFilePart != null) ? uploadedFilePart.getFile() : null;
        InputStream is = new FileInputStream(uploadedFile);
        
        Binary binary = session.getValueFactory().createBinary(is);
        node.setProperty(propertyName, binary);
    }
    
    private void _setReference(Session session, Node node, String propertyName, String[] value, boolean multiple) throws RepositoryException
    {
        List<Value> values = new ArrayList<>();
        ValueFactory vf = session.getValueFactory();
        for (String v : value)
        {
            values.add(vf.createValue(session.getNodeByIdentifier(v)));
        }
        
        boolean isMultiple = multiple || node.hasProperty(propertyName) && node.getProperty(propertyName).getDefinition().isMultiple();
        if (isMultiple)
        {
            node.setProperty(propertyName, values.toArray(new Value[values.size()]));
        }
        else
        {
            node.setProperty(propertyName, values.get(0));
        }
    }
    
    private void _setValue(Node node, String propertyName, String[] value, boolean multiple, int jcrType) throws RepositoryException
    {
        boolean isMultiple = multiple || node.hasProperty(propertyName) && node.getProperty(propertyName).getDefinition().isMultiple();
        if (isMultiple)
        {
            node.setProperty(propertyName, value, jcrType);
        }
        else
        {
            node.setProperty(propertyName, value[0], jcrType);
        }
    }
    
}
