/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repositoryapp.jcr;

import java.util.Map;
import java.util.Map.Entry;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.version.Version;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;

import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.core.NodeImpl;
import org.slf4j.Logger;

/**
 * Helper to manipulate the JCR version history
 */
public final class VersionHistoryHelper
{
    /**
     * Result of a remove version history operation
     */
    public enum RemoveHistoryResult
    {
        /** The version history has been removed */
        REMOVED,
        /** The version history is still referenced and could not be removed */
        REFERENCED,
        /** An error occurred preventing the removal */
        ERROR,
        /** The version history was broken and has been removed */
        FIXED,
        /** The version history is broken but could not be removed */
        UNFIXABLE
    }
    
    private VersionHistoryHelper()
    {
        // Utility class
    }
    
    /**
     * Remove the version history if it is unused (ie the versionable node does not exist anymore)
     * @param sessions the sessions used to determine if the versionable exist. Multiple session can be provided to search in multiple workspace for example.
     * @param versionHistory the version history node to remove
     * @param logger a logger to use to provide detailed information
     * @return the result of the operation
     * @throws RepositoryException if an error occurs
     */
    public static RemoveHistoryResult removeUnusedHistory(Map<String, Session> sessions, VersionHistory versionHistory, Logger logger) throws RepositoryException
    {
        String versionableIdentifier = versionHistory.getVersionableIdentifier();
        
        // check that the history is actually unused
        for (Entry<String, Session> entry: sessions.entrySet())
        {
            try
            {
                entry.getValue().getNodeByIdentifier(versionableIdentifier);
                return RemoveHistoryResult.REFERENCED;
            }
            catch (RepositoryException e)
            {
                // nothing
            }
        }
        
        // unused node, we may remove associated version history
        VersionIterator it = versionHistory.getAllVersions();
        
        int childNodesSize = 0;
        
        while (it.hasNext())
        {
            Version version = it.nextVersion();
            if (!JcrConstants.JCR_ROOTVERSION.equals(version.getName()))
            {
                childNodesSize++;
                
                try
                {
                    versionHistory.removeVersion(version.getName());
                }
                catch (RepositoryException e)
                {
                    logger.error("Error with version " + version.getName() + " of " + versionHistory.getIdentifier() + "... " + e);
                    return RemoveHistoryResult.ERROR;
                }
            }
        }
        
        if (childNodesSize == 0)
        {
            _removeEmptyHistory(versionHistory, sessions.get("default"));
            try
            {
                sessions.get("default").getNodeByIdentifier(versionHistory.getIdentifier());
                
                logger.debug("Empty history node is " + versionHistory.getIdentifier());
                return RemoveHistoryResult.UNFIXABLE;
            }
            catch (RepositoryException e)
            {
                return RemoveHistoryResult.FIXED;
            }
        }
        else
        {
            return RemoveHistoryResult.REMOVED;
        }
    }
    
    private static void _removeEmptyHistory(VersionHistory vh, Session session) throws RepositoryException
    {
        var node = ((NodeImpl) session.getNode("/ametys:root")).addNodeWithUuid("historyrepair", JcrConstants.NT_UNSTRUCTURED, vh.getProperty(JcrConstants.JCR_VERSIONABLEUUID).getString());
        node.addMixin("mix:versionable");
        session.save();
        session.getWorkspace().getVersionManager().checkin(node.getPath());
        session.getWorkspace().getVersionManager().checkout(node.getPath());
        session.removeItem("/ametys:root/historyrepair");
        session.save();
        vh.removeVersion("1.0");
    }

}
