/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.Repository;
import javax.jcr.SimpleCredentials;
import javax.naming.InitialContext;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.jackrabbit.core.TransientRepository;
import org.apache.jackrabbit.core.config.RepositoryConfig;

import org.ametys.core.util.SystemStatus;
import org.ametys.plugins.repositoryapp.RepositoryProvider;
import org.ametys.workspaces.repository.maintenance.MaintenanceTaskManager;

/**
 * Save a session to access to an external repository
 */
public class ConnectAction extends ServiceableAction implements ThreadSafe
{
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    /** System status provider */
    protected SystemStatus _systemStatus;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _repositoryProvider = (RepositoryProvider) serviceManager.lookup(RepositoryProvider.ROLE);
        _systemStatus = (SystemStatus) serviceManager.lookup(SystemStatus.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver sourceResolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        if (_repositoryProvider.getRepository() != null)
        {
            // The repository is already registered: return null.
            return null;
        }
        
        // Check system status, if state = repository unavailable, a maintenance
        // task is running, connection is not requested
        if (_systemStatus.getStatus().contains(MaintenanceTaskManager.REPOSITORY_UNAVAILABLE))
        {
            return null;
        }
        
        String mode = request.getParameter("mode");
        if (mode == null)
        {
            // The repository is not registered and the user didn't try to connect: display the form.
            result.put("result", "form");
            return result;
        }
        
        try
        {
            Repository repository = null;
            
            boolean jndi = "jndi".equals(mode);
            
            if (!jndi)
            {
                // jackrabbit connection mode
                String path = request.getParameter("repository-path");
                String configFile = request.getParameter("repository-xml");
                
                // Create repository
                RepositoryConfig config = RepositoryConfig.create(configFile, path);
                repository = new TransientRepository(config);
            }
            else
            {
                // jndi connection mode
                String jndiContext = request.getParameter("jndi-context");
                
                InitialContext ctx = new InitialContext();
                repository = (Repository) ctx.lookup(jndiContext);
            }
            
            // Credentials
            String user = request.getParameter("user");
            String pwd = request.getParameter("password");
            
            SimpleCredentials credentials = new SimpleCredentials(user, pwd.toCharArray());
            
            // Register the repository and initialize the session on the "default" workspace.
            _repositoryProvider.registerRepository(repository, credentials, jndi);
            _repositoryProvider.getSession("default");
            
            return null;
        }
        catch (Exception e)
        {
            getLogger().error("Error connecting to the repository.", e);
            
            result.put("result", "error");
            result.put("error", "connection");
            return result;
        }
    }
}
