/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository;

import java.io.IOException;

import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.repositoryapp.RepositoryProvider;

/**
 * Generate descriptors of the repository
 */
public class DescriptorsGenerator extends ServiceableGenerator
{
    
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _repositoryProvider = (RepositoryProvider) serviceManager.lookup(RepositoryProvider.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        // get current repository
        Repository repository = _repositoryProvider.getRepository();

        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "repository");
        
        String[] descriptorKeys = repository.getDescriptorKeys();
        for (String key : descriptorKeys)
        {
            Value[] values = repository.getDescriptorValues(key);
            
            AttributesImpl descAttrs = new AttributesImpl();
            descAttrs.addCDATAAttribute("key", key);
            descAttrs.addCDATAAttribute("standard", String.valueOf(repository.isStandardDescriptor(key)));
            descAttrs.addCDATAAttribute("singleValue", String.valueOf(repository.isSingleValueDescriptor(key)));
            
            XMLUtils.startElement(contentHandler, "descriptor", descAttrs);
            XMLUtils.startElement(contentHandler, "values");
            
            if (values != null)
            {
                for (Value value : values)
                {
                    try
                    {
                        AttributesImpl attrs = new AttributesImpl();
                        attrs.addCDATAAttribute("value", value.getString());
//                        attrs.addCDATAAttribute("type", PropertyType.nameFromValue(value.getType()));
//                        attrs.addCDATAAttribute("typeCode", String.valueOf(value.getType()));
                        XMLUtils.createElement(contentHandler, "value", attrs);
                    }
                    catch (RepositoryException e)
                    {
                        getLogger().error("Error while retrieving a repository descriptor.", e);
                    }
                }
            }
            
            XMLUtils.endElement(contentHandler, "values");
            XMLUtils.endElement(contentHandler, "descriptor");
        }
        
        XMLUtils.endElement(contentHandler, "repository");
        
        contentHandler.endDocument();
    }
    
}
