/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.Reader;
import org.apache.cocoon.reading.ServiceableReader;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.JcrConstants;
import org.xml.sax.SAXException;

import org.ametys.plugins.repositoryapp.RepositoryProvider;

/**
 * {@link Reader} for exposing a binary property from a JCR repository.<br>
 * Mime type can be forced by using parameter <code>mime-type</code>.
 */
public class BinaryPropertyReader extends ServiceableReader
{
    
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    private Node _node;
    private Property _property;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _repositoryProvider = (RepositoryProvider) serviceManager.lookup(RepositoryProvider.ROLE);
    }
    
    @Override
    public void setup(SourceResolver res, Map objModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        super.setup(res, objModel, src, par);
        
        Session session = null;
        String workspaceName = null;
        
        try
        {
            workspaceName = parameters.getParameter("workspace");
            
            session = _repositoryProvider.getSession(workspaceName);
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to retrieve repository", e);
        }
        
        try
        {
            String path = par.getParameter("path", "");
            String relPath = RepositoryDao.removeLeadingSlash(path);
            
            _node = session.getRootNode().getNode(relPath);
            
            String name = par.getParameter("property", null);
            _property = _node.getProperty(name);
        }
        catch (RepositoryException e)
        {
            throw new ProcessingException(e);
        }
    }
    
    @Override
    public String getMimeType()
    {
        try
        {
            if (_node.hasProperty(JcrConstants.JCR_MIMETYPE))
            {
                return _node.getProperty(JcrConstants.JCR_MIMETYPE).getString();
            }
        }
        catch (RepositoryException e)
        {
            // Nothing
        }
        
        return "application/unknown";
    }
    
    private String _getName () throws RepositoryException
    {
        if (JcrConstants.NT_RESOURCE.equals(_node.getPrimaryNodeType().getName()))
        {
            return _node.getParent().getName();
        }
        
        return _property.getName();
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        String path = null;
        try
        {
            String name = _getName ();
            name = name.replaceAll("\\\\", "\\\\\\\\");
            name = name.replaceAll("\\\"", "\\\\\\\"");
            
            path = _property.getPath();
            
            Response response = ObjectModelHelper.getResponse(objectModel);
            
            response.setHeader("Content-Length", Long.toString(_property.getLength()));
            response.setHeader("Content-Disposition", "attachment; filename=\"" + name + "\"");
            
            try (InputStream is = _property.getBinary().getStream())
            {
                IOUtils.copy(is, out);
            }
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to download binary property of path " + path, e);
        }
        finally
        {
            IOUtils.closeQuietly(out);
        }
    }
}
