/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

/**
 * Node group helper.
 */
public final class NodeGroupHelper
{
    
    /** The maximum children to display per node. */
    public static final int MAX_CHILDREN_PER_NODE = 100;
    
    private NodeGroupHelper()
    {
        // Hides the default constructor.
    }
    
    /**
     * Get the node's full path, including virtual path segments.
     * @param node the node of which to get the path.
     * @return the computed path.
     * @throws RepositoryException if a repository error occurs.
     */
    public static String getPathWithGroups(Node node) throws RepositoryException
    {
        StringBuffer fullPath = new StringBuffer();
        
        // Compute and append all ancestor paths.
        int depth = node.getDepth();
        for (int i = 0; i < depth; i++)
        {
            Node ancestor = (Node) node.getAncestor(i);
            Node ancestorChild = (Node) node.getAncestor(i + 1);
            
            if (!ancestor.getName().equals(""))
            {
                fullPath.append('/').append(ancestor.getName());
                
                int index = ancestor.getIndex();
                if (index > 1)
                {
                    fullPath.append('[').append(index).append(']');
                }
            }
            
            // Append the virtual paths if needed.
            fullPath.append(getGroupPaths(ancestorChild, MAX_CHILDREN_PER_NODE));
        }
        
        // Append the node path itself.
        fullPath.append('/').append(node.getName());
        
        int index = node.getIndex();
        if (index > 1)
        {
            fullPath.append('[').append(index).append(']');
        }
        
        return fullPath.toString();
    }
    
    /**
     * Get the virtual a node's path relative to its parent.
     * If the parent node has more than a specified node count, "virtual" paths
     * are added between the parent and the node.
     * The empty string is returned if the parent doesn't have more nodes than the maximum.
     * @param node the node to get the virtual path of.
     * @param maxNodesPerGroup the maximum node per group.
     * @return the virtual path part of the node in its parent.
     * @throws RepositoryException if a repository error occurs.
     */
    private static String getGroupPaths(Node node, int maxNodesPerGroup) throws RepositoryException
    {
        StringBuilder nodePath = new StringBuilder();
        
        NodeIterator children = node.getParent().getNodes();
        long childCount = children.getSize();
        if (childCount > maxNodesPerGroup)
        {
            // Compute virtual path.
            int level = (int) Math.ceil(Math.log(childCount) / Math.log(maxNodesPerGroup)) - 1;
            
            int position = 0;
            boolean found = false;
            while (children.hasNext() && !found)
            {
                if (children.nextNode().isSame(node))
                {
                    found = true;
                }
                else
                {
                    position++;
                }
            }
            
            // The position has been found.
            for (int j = level; j > 0; j--)
            {
                int currentGroupCount = (int) Math.pow(maxNodesPerGroup, j);
                int groupIndex = position / currentGroupCount;
                
                int startPos = groupIndex * currentGroupCount + 1;
                int endPos = Math.min(startPos + currentGroupCount - 1, (int) childCount);
                
                nodePath.append('/').append(startPos).append("...").append(endPos);
            }
        }
        
        return nodePath.toString();
    }
    
}
