/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.workspaces.repository.jcr;

import java.util.NoSuchElementException;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

/**
 * Provides helper methods on nodes.
 */
public final class NodeHelper
{
    private NodeHelper()
    {
        // Hides the default constructor.
    }

    /**
     * Rename the given {@link Node} with the given new name
     * @param node the node to rename
     * @param newName the new name of the node
     * @throws RepositoryException if an error occurs.
     */
    public static void rename(Node node, String newName) throws RepositoryException
    {
        Node parentNode = node.getParent();
        boolean order = parentNode.getPrimaryNodeType().hasOrderableChildNodes();
        Node nextSibling = null;
        
        if (order)
        {
            // iterate over the siblings to find the following
            NodeIterator siblings = parentNode.getNodes();
            boolean iterate = true;
            
            while (siblings.hasNext() && iterate)
            {
                Node sibling = siblings.nextNode();
                iterate = !sibling.getName().equals(node.getName());
            }
            
            // iterator is currently on the node
            while (siblings.hasNext() && nextSibling == null)
            {
                Node sibling = siblings.nextNode();
                String path = sibling.getPath();
                if (node.getSession().itemExists(path))
                {
                    nextSibling = sibling;
                }
            }
        }
        
        node.getSession().move(node.getPath(), node.getParent().getPath() + "/" + newName);
        
        if (order)
        {
            // nextSibling is either null meaning that the Node must be ordered last or is equals to the following sibling
            if (nextSibling != null)
            {
                parentNode.orderBefore(newName, nextSibling.getName());
            }
            else
            {
                parentNode.orderBefore(newName, null);
            }
        }
    }
    
    /**
     * Get the child node at the given index
     * @param node The parent node
     * @param index The position of the child node
     * @return The child node
     * @throws NoSuchElementException If no node exist
     * @throws RepositoryException If an error occurred
     */
    public static Node getChildAt(Node node, int index) throws NoSuchElementException, RepositoryException
    {
        if (index < 0)
        {
            throw new NoSuchElementException("Child node index cannot be negative (" + index + ")");
        }
        
        NodeIterator nodes = node.getNodes();
        nodes.skip(index);
        return nodes.nextNode();
    }
}
