/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;

/**
 * Node state tracker.
 */
public class NodeStateTracker extends AbstractLogEnabled implements Component, Initializable
{
    
    /** The avalon role. */
    public static final String ROLE = NodeStateTracker.class.getName();
    
    /** The nodes created but not already saved. */
    protected Map<String, Set<String>> _newNodes;
    
    @Override
    public void initialize() throws Exception
    {
        _newNodes = new HashMap<>();
    }
    
    /**
     * Get the new nodes in a workspace.
     * @param workspaceName the workspace name.
     * @return the newly created nodes.
     */
    public Set<String> getNewNodes(String workspaceName)
    {
        Set<String> nodes = new HashSet<>();
        
        if (_newNodes.containsKey(workspaceName))
        {
            nodes.addAll(_newNodes.get(workspaceName));
        }
        
        return nodes;
    }
    
    /**
     * A node was added.
     * @param workspaceName the workspace name.
     * @param path the added node path.
     * @throws RepositoryException if an error occurred
     */
    public void nodeAdded(String workspaceName, String path) throws RepositoryException
    {
        Set<String> newNodes = null;
        if (_newNodes.containsKey(workspaceName))
        {
            newNodes = _newNodes.get(workspaceName);
        }
        else
        {
            newNodes = new HashSet<>();
            _newNodes.put(workspaceName, newNodes);
        }
        
        newNodes.add(path);
    }
    
    /**
     * A node was removed.
     * @param workspaceName the workspace name.
     * @param path the removed node path.
     * @throws RepositoryException if an error occurred
     */
    public void nodeRemoved(String workspaceName, String path) throws RepositoryException
    {
        Set<String> newNodes = null;
        if (_newNodes.containsKey(workspaceName))
        {
            newNodes = _newNodes.get(workspaceName);
        }
        else
        {
            newNodes = new HashSet<>();
            _newNodes.put(workspaceName, newNodes);
        }
        
        newNodes.remove(path);
    }
    
    /**
     * Clear the node set for a workspace.
     * @param workspaceName the workspace name.
     */
    public void clear(String workspaceName)
    {
        if (_newNodes.containsKey(workspaceName))
        {
            _newNodes.get(workspaceName).clear();
        }
    }
    
}
