/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Workspace;
import javax.jcr.nodetype.NodeDefinition;
import javax.jcr.nodetype.NodeType;
import javax.jcr.nodetype.NodeTypeIterator;
import javax.jcr.nodetype.NodeTypeManager;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.plugins.repositoryapp.RepositoryProvider;

/**
 * Node type hierarchy component.
 */
public class NodeTypeHierarchyComponent extends AbstractLogEnabled implements Component, Contextualizable, Serviceable
{
    
    /** The avalon role. */
    public static final String ROLE = NodeTypeHierarchyComponent.class.getName();
    
    /** The subtypes registry, by workspace. */
    protected Map<String, Map<String, Set<String>>> _subTypes;
    
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    /** The avalon context. */
    protected Context _context;
    
    /** The service manager. */
    protected ServiceManager _manager;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        this._context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _repositoryProvider = (RepositoryProvider) manager.lookup(RepositoryProvider.ROLE);
    }
    
    /**
     * Get the sub types of a given node type in a specific workspace.
     * @param nodeType the node type.
     * @param workspaceName the workspace name.
     * @return the sub types of the node type in the given workspace.
     * @throws RepositoryException if an error occurred
     */
    public Set<String> getDirectSubTypes(NodeType nodeType, String workspaceName) throws RepositoryException
    {
        if (_subTypes == null || !_subTypes.containsKey(workspaceName))
        {
            _buildTypeHierarchy(workspaceName);
        }
        
        if (_subTypes.get(workspaceName).containsKey(nodeType.getName()))
        {
            return Collections.unmodifiableSet(_subTypes.get(workspaceName).get(nodeType.getName()));
        }
        else
        {
            return Collections.emptySet();
        }
    }
    
    /**
     * Get the sub types of a given node type in a specific workspace.
     * @param nodeType the node type.
     * @param workspaceName the workspace name.
     * @return the sub types of the node type in the given workspace.
     * @throws RepositoryException if an error occurred
     */
    public Set<String> getRecursiveSubTypes(String nodeType, String workspaceName) throws RepositoryException
    {
        if (_subTypes == null || !_subTypes.containsKey(workspaceName))
        {
            _buildTypeHierarchy(workspaceName);
        }
        
        Set<String> allTypes = new HashSet<>();
        
        if (_subTypes.get(workspaceName).containsKey(nodeType))
        {
            for (String subType : _subTypes.get(workspaceName).get(nodeType))
            {
                allTypes.add(subType);
                allTypes.addAll(getRecursiveSubTypes(subType, workspaceName));
            }
        }
        
        return allTypes;
    }
    
    /**
     * Get the available children types for a given node type in a workspace.
     * @param nodeDef the node definition.
     * @param workspaceName the workspace name.
     * @return the available node types.
     * @throws RepositoryException if an error occurred
     */
    public Set<String> getAvailableChildrenTypes(NodeDefinition nodeDef, String workspaceName) throws RepositoryException
    {
        if (_subTypes == null || !_subTypes.containsKey(workspaceName))
        {
            _buildTypeHierarchy(workspaceName);
        }
        
        Set<String> types = new HashSet<>();
        
        NodeType[] requiredTypes = nodeDef.getRequiredPrimaryTypes();
        if (requiredTypes.length > 0)
        {
            types.add(requiredTypes[0].getName());
            types.addAll(getRecursiveSubTypes(requiredTypes[0].getName(), workspaceName));
        }
        
        for (int i = 1; i < requiredTypes.length; i++)
        {
            types.add(requiredTypes[0].getName());
            types.retainAll(getRecursiveSubTypes(requiredTypes[0].getName(), workspaceName));
        }
        
        return types;
    }
    
    /**
     * Compute the primary type hierarchy.
     * @param workspaceName The workspace name to use
     * @throws RepositoryException if an error occurs.
     */
    protected void _buildTypeHierarchy(String workspaceName) throws RepositoryException
    {
        if (_subTypes == null)
        {
            _subTypes = new HashMap<>();
        }
        
        Map<String, Set<String>> workspaceTypes = new HashMap<>();
        
        _subTypes.put(workspaceName, workspaceTypes);
        
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            Workspace workspace = _getWorkspace(request, workspaceName);
            NodeTypeManager ntManager = workspace.getNodeTypeManager();
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Computing node type hierarchy for workspace " + workspaceName);
            }
            
            NodeTypeIterator nodeTypes = ntManager.getPrimaryNodeTypes();
            while (nodeTypes.hasNext())
            {
                NodeType nodeType = nodeTypes.nextNodeType();
                for (NodeType supertype : nodeType.getSupertypes())
                {
                    if (!supertype.isMixin())
                    {
                        Set<String> subtypes = null;
                        if (workspaceTypes.containsKey(supertype.getName()))
                        {
                            subtypes = workspaceTypes.get(supertype.getName());
                        }
                        else
                        {
                            subtypes = new HashSet<>();
                            workspaceTypes.put(supertype.getName(), subtypes);
                        }
                        
                        if (getLogger().isDebugEnabled())
                        {
                            getLogger().debug(nodeType.getName() + " is a sub type of " + supertype.getName());
                        }
                        
                        subtypes.add(nodeType.getName());
                    }
                }
            }
        }
        catch (ServiceException e)
        {
            throw new RuntimeException(e);
        }
    }
    
    /**
     * Get the repository and session and set the _repository and _session protected fields
     * @param request The request
     * @param workspaceName The jcr workspace
     * @return the workspace.
     * @throws RepositoryException if an error occured.
     * @throws ServiceException if the repository can't be looked up.
     */
    protected Workspace _getWorkspace(Request request, String workspaceName) throws RepositoryException, ServiceException
    {
        Session session = _repositoryProvider.getSession(workspaceName);
        
        return session.getWorkspace();
    }
    
}
