/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.io.IOException;

import javax.jcr.Node;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;
import javax.jcr.query.Row;
import javax.jcr.query.RowIterator;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

/**
 * Generate a list of path from an XPath query
 */
public class QueryResultGenerator extends AbstractRepositoryGenerator
{
    @SuppressWarnings("deprecation")
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String workspaceName = parameters.getParameter("workspace", "default");
        
        contentHandler.startDocument();
        
        if (StringUtils.isNotBlank(workspaceName))
        {
            _getRepository(request, workspaceName);
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Trying to generate results for a XPath query");
            }
            
            XMLUtils.startElement(contentHandler, "resultList");
            
            String query = parameters.getParameter("query", "");
            String language = parameters.getParameter("language", Query.XPATH);
            
            if (StringUtils.isNotBlank(query))
            {
                _populateQueryResult(_session, query, language);
            }
            
            XMLUtils.endElement(contentHandler, "resultList");
        }
        
        contentHandler.endDocument();
    }


    private void _populateQueryResult(Session session, String query, String language) throws SAXException
    {
        try
        {
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            QueryResult queryResult = queryManager.createQuery(query, language).execute();
            String[] selectorNames = queryResult.getSelectorNames();
            String selector = selectorNames.length > 0 ? selectorNames[0] : null;
            
            RowIterator rows = queryResult.getRows();
           
            while (rows.hasNext())
            {
                Row row = rows.nextRow();
                Node node = selector == null ? row.getNode() : row.getNode(selector);
                
                // Node can be null when querying with outer joins.
                if (node != null)
                {
                    AttributesImpl attr = new AttributesImpl();
                    attr.addCDATAAttribute("path", node.getPath());
                    attr.addCDATAAttribute("pathWithGroups", NodeGroupHelper.getPathWithGroups(node));
                    attr.addCDATAAttribute("name", node.getName());
                    XMLUtils.createElement(contentHandler, "result", attr);
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Unable to execute query '" + query + "'", e);
            XMLUtils.createElement(contentHandler, "error", e.toString());
        }
    }
    
}
