/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.UnsupportedRepositoryOperationException;
import javax.jcr.Workspace;
import javax.jcr.lock.LockManager;
import javax.jcr.nodetype.NodeDefinition;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.repositoryapp.RepositoryProvider;
import org.ametys.runtime.config.Config;

/**
 * Component providing methods to access the repository.
 */
public class RepositoryDao extends AbstractLogEnabled implements Component, Serviceable
{
    
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    /** The node state tracker. */
    protected NodeStateTracker _nodeStateTracker;
    
    /** The node type hierarchy component. */
    protected NodeTypeHierarchyComponent _nodeTypeHierarchy;
    
    private ServiceManager _serviceManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _serviceManager = serviceManager;
        _repositoryProvider = (RepositoryProvider) serviceManager.lookup(RepositoryProvider.ROLE);
        _nodeStateTracker = (NodeStateTracker) serviceManager.lookup(NodeStateTracker.ROLE);
        _nodeTypeHierarchy = (NodeTypeHierarchyComponent) serviceManager.lookup(NodeTypeHierarchyComponent.ROLE);
    }
    
    /**
     * Get information on the repository.
     * @return information on the repository as a Map.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getRepositoryInfo()
    {
        Map<String, Object> info = new HashMap<>();
        
        info.put("standalone", _serviceManager.hasService(Repository.class.getName()));
        
        // Must be compatible with safe mode
        Config configInstance = Config.getInstance();
        if (configInstance != null)
        {
            String defaultOrder = configInstance.getValue("repository.default.sort");
            info.put("defaultOrder", defaultOrder);
        }
        
        return info;
    }
    
    /**
     * Get the list of available workspaces in the repository.
     * @return the list of available workspaces in the repository.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public List<String> getWorkspaces() throws RepositoryException
    {
        Session session = _repositoryProvider.getSession("default");
        
        Workspace workspace = session.getWorkspace();
        
        return Arrays.asList(workspace.getAccessibleWorkspaceNames());
    }
    
    /**
     * Get node information by path.
     * @param paths the node paths, relative to the root node (without leading slash).
     * @param workspaceName the workspace name.
     * @return information on the node as a Map.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getNodesByPath(Collection<String> paths, String workspaceName) throws RepositoryException
    {
        Session session = _repositoryProvider.getSession(workspaceName);
        Node rootNode = session.getRootNode();
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        List<String> notFound = new ArrayList<>();
        
        for (String path : paths)
        {
            try
            {
                Node node = null;
                
                String relPath = removeLeadingSlash(path);
                if (StringUtils.isEmpty(relPath))
                {
                    node = rootNode;
                }
                else
                {
                    node = rootNode.getNode(relPath);
                }
                
                Map<String, Object> nodeInfo = new HashMap<>();
                fillNodeInfo(node, nodeInfo);
                nodes.add(nodeInfo);
            }
            catch (PathNotFoundException e)
            {
                notFound.add(path);
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("nodes", nodes);
        result.put("notFound", notFound);
        
        return result;
    }
    
    /**
     * Get node information by path.
     * @param path the node path, relative to the root node (without leading slash).
     * @param workspaceName the workspace name.
     * @return information on the node as a Map.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getNodeByPath(String path, String workspaceName) throws RepositoryException
    {
        Session session = _repositoryProvider.getSession(workspaceName);
        String relPath = removeLeadingSlash(path);
        Node node = session.getRootNode().getNode(relPath);
        
        Map<String, Object> nodeInfo = new HashMap<>();
        
        fillNodeInfo(node, nodeInfo);
        
        return nodeInfo;
    }
    
    /**
     * Get node information by its identifier.
     * @param identifier the node identifier.
     * @param workspaceName the workspace name.
     * @return information on the node as a Map.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getNodeByIdentifier(String identifier, String workspaceName) throws RepositoryException
    {
        Session session = _repositoryProvider.getSession(workspaceName);
        try
        {
            Node node = session.getNodeByIdentifier(identifier);
            
            Map<String, Object> nodeInfo = new HashMap<>();
            
            fillNodeInfo(node, nodeInfo);
            
            return nodeInfo;
        }
        catch (ItemNotFoundException e)
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn(String.format("Item '%s' not found in workspace '%s'", identifier, workspaceName), e);
            }
            
            return null;
        }
    }
    
    /**
     * Get the possible children types of a node.
     * @param nodePath The node path.
     * @param workspaceName The workspace name.
     * @return the possible children types of the node.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Set<String> getChildrenTypes(String nodePath, String workspaceName) throws RepositoryException
    {
        Session session = _repositoryProvider.getSession(workspaceName);
        
        String relPath = removeLeadingSlash(nodePath);
        
        Node node = null;
        if (StringUtils.isEmpty(relPath))
        {
            node = session.getRootNode();
        }
        else
        {
            node = session.getRootNode().getNode(relPath);
        }
    
        // Store allowed types in this set
        Set<String> availableChildrenTypes = new HashSet<>();
        NodeDefinition[] childNodeDefinitions = node.getPrimaryNodeType().getChildNodeDefinitions();
        for (NodeDefinition nodeDef : childNodeDefinitions)
        {
            availableChildrenTypes.addAll(_nodeTypeHierarchy.getAvailableChildrenTypes(nodeDef, workspaceName));
        }
        
        return availableChildrenTypes;
    }
    
    /**
     * Add a node.
     * @param parentPath the parent node path.
     * @param childName the name of the node to create.
     * @param childType the type of the node to create.
     * @param workspaceName the workspace name.
     * @return A Map with information on the created node.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> addNode(String parentPath, String childName, String childType, String workspaceName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to add child: '" + childName + "' to the node at path: '" + parentPath + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        String relPath = removeLeadingSlash(parentPath);
        
        // Get the parent node
        Node parentNode = null;
        if (StringUtils.isEmpty(relPath))
        {
            parentNode = session.getRootNode();
        }
        else
        {
            parentNode = session.getRootNode().getNode(relPath);
        }
        
        // Add the new child to the parent
        Node childNode = parentNode.addNode(childName, childType);
        
        String fullPath = NodeGroupHelper.getPathWithGroups(childNode);
        
        Map<String, Object> result = new HashMap<>();
        result.put("path", childNode.getPath());
        result.put("pathWithGroups", fullPath);
        
        _nodeStateTracker.nodeAdded(workspaceName, fullPath);
        
        return result;
    }
    
    /**
     * Remove a node from the repository.
     * @param path The absolute node path, can start with a slash or not. 
     * @param workspaceName The workspace name.
     * @return The full parent path.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public String removeNode(String path, String workspaceName) throws RepositoryException
    {
        Session session = _repositoryProvider.getSession(workspaceName);
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to remove node at path: '" + path + "'");
        }
        
        String relPath = removeLeadingSlash(path);
        
        // Get and remove the node.
        Node node = session.getRootNode().getNode(relPath);
        Node parentNode = node.getParent();
        
        String fullPath = NodeGroupHelper.getPathWithGroups(node);
        String fullParentPath = NodeGroupHelper.getPathWithGroups(parentNode);
        
        node.remove();
        
        _nodeStateTracker.nodeRemoved(workspaceName, fullPath);
        _nodeStateTracker.nodeAdded(workspaceName, fullParentPath);
        
        return fullParentPath;
    }
    
    /**
     * Remove a property from a node.
     * @param path The absolute node path, can start with a slash or not.
     * @param workspaceName The workspace name.
     * @param propertyName The name of the property to remove.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void removeProperty(String path, String workspaceName, String propertyName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Removing property '" + propertyName + "' from the node at path '" + path + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        String relPath = removeLeadingSlash(path);
        
        Node node = null;
        if (StringUtils.isEmpty(relPath))
        {
            node = session.getRootNode();
        }
        else
        {
            node = session.getRootNode().getNode(relPath);
        }

        // Remove the property
        node.getProperty(propertyName).remove();
    }
    
    /**
     * Unlock a node.
     * @param path The absolute node path.
     * @param workspaceName The workspace name.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void unlockNode(String path, String workspaceName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to unlock the node at path '" + path + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        LockManager lockManager = session.getWorkspace().getLockManager();
        
        try
        {
            // Try to add lock token stored on AmetysObject
            Node node = session.getNode(path);
            if (node.hasProperty("ametys-internal:lockToken"))
            {
                String lockToken = node.getProperty("ametys-internal:lockToken").getString();
                lockManager.addLockToken(lockToken);
            }
            else if (getLogger().isInfoEnabled())
            {
                getLogger().info("Lock token property not found for node at path '" + path + "'");
            }
        }
        catch (RepositoryException e)
        {
            getLogger().warn("Unable to add locken token to unlock node at path '" + path + "'", e);
        }
        
        session.getWorkspace().getLockManager().unlock(path);
    }
    
    /**
     * Check-out a node.
     * @param path The absolute node path, must start with a slash.
     * @param workspaceName The workspace name.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void checkoutNode(String path, String workspaceName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to checkout the node at path '" + path + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        
        // Check the node out.
        session.getWorkspace().getVersionManager().checkout(path);
    }
    
    /**
     * Change the name of a node
     * @param path The absolute node path, must start with a slash.
     * @param workspaceName The workspace name.
     * @param newName The new name of the node in the same parent
     * @return A Map with information on the renamed node.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> renameNode(String path, String workspaceName, String newName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to rename the node at path '" + path + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        
        String relPath = removeLeadingSlash(path);
        
        Node node = session.getRootNode().getNode(relPath);
        
        String fullPathBefore = NodeGroupHelper.getPathWithGroups(node);
        NodeHelper.rename(node, newName);
        String fullPathAfter = NodeGroupHelper.getPathWithGroups(node);
        
        Map<String, Object> result = new HashMap<>();
        result.put("path", node.getPath());
        result.put("pathWithGroups", fullPathAfter);
        
        _nodeStateTracker.nodeRemoved(workspaceName, fullPathBefore);
        _nodeStateTracker.nodeAdded(workspaceName, fullPathAfter);
        
        return result;
    }
    
    /**
     * Change the name of a node
     * @param paths The absolute source node paths, must start with a slash.
     * @param workspaceName The workspace name.
     * @param targetPath The absolute target node paths, must start with a slash.
     * @param index The insert position in child nodes. -1 means as last child.
     * @return A Map with information on the moved node.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, List<String>> moveNodes(List<String> paths, String workspaceName, String targetPath, int index) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to move the node at path '" + targetPath + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);

        String relTargetPath = removeLeadingSlash(targetPath);
        Node targetNode = session.getRootNode().getNode(relTargetPath);

        Map<String, List<String>> result = new HashMap<>();
        result.put("path", new ArrayList<>());
        result.put("pathWithGroups", new ArrayList<>());

        for (String path : paths)
        {
            String relPath = removeLeadingSlash(path);
            
            Node node = session.getRootNode().getNode(relPath);
            Node parentNode = node.getParent();
    
            String fullPathBefore = NodeGroupHelper.getPathWithGroups(node);
            String fullParentPathBefore = NodeGroupHelper.getPathWithGroups(parentNode);
            String fullTargetPath = NodeGroupHelper.getPathWithGroups(targetNode);
            
            if (!fullParentPathBefore.equals(fullTargetPath))
            {
                // Moving to another parent
                session.move(fullPathBefore, fullTargetPath + "/" + node.getName());
            }
            else if (index == -1)
            {
                // Same parent (last child)
                // Ordering may be forbidden and will throw an error
                parentNode.orderBefore(StringUtils.substringAfterLast(fullPathBefore, "/"), null);
            }
            
            if (index != -1)
            {
                // Reordering in the parent (can be a new or the old parent)
                String fullPathBefore2 = NodeGroupHelper.getPathWithGroups(node);
                Node parentNode2 = node.getParent();
                Node brother = NodeHelper.getChildAt(parentNode2, index);
                String brotherFullPath = NodeGroupHelper.getPathWithGroups(brother);
                try
                {
                    parentNode2.orderBefore(StringUtils.substringAfterLast(fullPathBefore2, "/"), StringUtils.substringAfterLast(brotherFullPath, "/"));
                }
                catch (UnsupportedRepositoryOperationException e)
                {
                    // Ordering may be forbidden and will throw an error
                    // If we effectively moved the node, let's catch the error... not that important + the node has change and we need to reflect it
                    if (fullPathBefore2.equals(fullPathBefore))
                    {
                        throw e;
                    }
                    getLogger().warn("While moving the node " + fullPathBefore + " to " + fullTargetPath + ", we could not place it at index " + index, e);
                }
            }
    
            String fullPathAfter = NodeGroupHelper.getPathWithGroups(node);
            
            result.get("path").add(node.getPath());
            result.get("pathWithGroups").add(fullPathAfter);
            
            _nodeStateTracker.nodeRemoved(workspaceName, fullPathBefore);
            _nodeStateTracker.nodeAdded(workspaceName, fullPathAfter);
        }
        
        return result;
    }
    
    /**
     * Save a session.
     * @param workspaceName The workspace name.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void saveSession(String workspaceName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Persisting session for workspace '" + workspaceName + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        
        session.save();
        
        _nodeStateTracker.clear(workspaceName);
    }
    
    /**
     * Rollback a session.
     * @param workspaceName The workspace name.
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public void rollbackSession(String workspaceName) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Rolling back session for workspace '" + workspaceName + "'");
        }
        
        Session session = _repositoryProvider.getSession(workspaceName);
        
        session.refresh(false);
        
        _nodeStateTracker.clear(workspaceName);
    }
    
    /**
     * Fill the node info.
     * @param node The node to convert
     * @param nodeInfo The map to fill
     * @throws RepositoryException if an error occurs getting or setting data from/in the repository.
     */
    protected void fillNodeInfo(Node node, Map<String, Object> nodeInfo) throws RepositoryException
    {
        boolean hasOrderableChildNodes = true;
        
        nodeInfo.put("id", node.getIdentifier());
        nodeInfo.put("path", node.getPath());
        nodeInfo.put("pathWithGroups", NodeGroupHelper.getPathWithGroups(node));
        nodeInfo.put("name", node.getName());
        nodeInfo.put("index", node.getIndex());
        nodeInfo.put("hasOrderableChildNodes", hasOrderableChildNodes);
        nodeInfo.put("locked", node.isLocked());
        nodeInfo.put("checkedOut", node.isCheckedOut());
    }
    
    /**
     * Add a leading slash to the path.
     * @param path the path.
     * @return the path with a leading slash.
     */
    public static String addLeadingSlash(String path)
    {
        if (StringUtils.isNotEmpty(path) && path.charAt(0) != '/')
        {
            return '/' + path;
        }
        return path;
    }
    
    /**
     * Remove the leading slash from the path if needed.
     * @param path the path.
     * @return the path without leading slash.
     */
    public static String removeLeadingSlash(String path)
    {
        if (StringUtils.isNotEmpty(path) && path.charAt(0) == '/')
        {
            return path.substring(1);
        }
        return path;
    }
}
