/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.runtime.config.Config;

/**
 * Generate the content of a node:
 * <ul>
 *  <li>properties
 *  <li>children
 *  <li>referers
 * </ul>
 */
public class RepositoryNodeGenerator extends AbstractRepositoryGenerator
{
    private static final Comparator<Node> __ALPHA_NODE_COMPARATOR = new NodeNameComparator();
    
    private static final Comparator<Node> __REVERSE_ALPHA_NODE_COMPARATOR = new NodeNameComparator(false);
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String workspaceName = parameters.getParameter("workspace", "default");
        // Can be "document", "alpha" or "reverseAlpha".
        String defaultOrder = Config.getInstance() != null ? Config.getInstance().getValue("repository.default.sort") : "document";
        String order = parameters.getParameter("order", defaultOrder);
        if (StringUtils.isBlank(order))
        {
            order = defaultOrder;
        }
            
        _getRepository(request, workspaceName);

        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Trying to generate content for a node");
        }
        
        contentHandler.startDocument();
        
        
        if (workspaceName != null)
        {
            AttributesImpl repositoryAttributes = new AttributesImpl();
            repositoryAttributes.addCDATAAttribute("workspace", workspaceName);
            XMLUtils.startElement(contentHandler, "repository", repositoryAttributes);
        }
        else
        {
            XMLUtils.startElement(contentHandler, "repository");
        }

        try
        {
            _saxNode(_session, order);
        }
        catch (RepositoryException e)
        {
            throw new ProcessingException("Unable to retrieve node for path: '" + source + "'", e);
        }
        
        XMLUtils.endElement(contentHandler, "repository");
        
        contentHandler.endDocument();
    }

    private void _saxNode(Session session, String order) throws SAXException, RepositoryException
    {
        Node node = null;
        Node rootNode = session.getRootNode();
        String path = parameters.getParameter("path", "");
        long start = parameters.getParameterAsLong("start", -1);
        long end = parameters.getParameterAsLong("end", -1);
        
        if (StringUtils.isEmpty(path))
        {
            node = rootNode;
        }
        else
        {
            node = rootNode.getNode(path);
        }
        
        AttributesImpl nodeAttributes = new AttributesImpl();
        
        if (rootNode != node)
        {
            nodeAttributes.addCDATAAttribute("parentPath", node.getParent().getPath());
        }
        else
        {
            // Parent path is empty if this is the JCR root node
            nodeAttributes.addCDATAAttribute("parentPath", "");
        }
        
        if (start > -1 && end > -1)
        {
            nodeAttributes.addCDATAAttribute("start", Long.toString(start));
            nodeAttributes.addCDATAAttribute("end", Long.toString(end));
        }
        
        boolean hasOrderableChildNodes = node.getPrimaryNodeType().hasOrderableChildNodes();
        
        nodeAttributes.addCDATAAttribute("name", node.getName());
        nodeAttributes.addCDATAAttribute("path", node.getPath());
        nodeAttributes.addCDATAAttribute("pathWithGroups", NodeGroupHelper.getPathWithGroups(node));
        nodeAttributes.addCDATAAttribute("index", String.valueOf(node.getIndex()));
        nodeAttributes.addCDATAAttribute("hasOrderableChildNodes", Boolean.toString(hasOrderableChildNodes));
        nodeAttributes.addCDATAAttribute("isNew", Boolean.toString(node.isNew()));
        nodeAttributes.addCDATAAttribute("isModified", Boolean.toString(node.isModified()));
        nodeAttributes.addCDATAAttribute("noChild", Boolean.toString(!node.hasNodes()));
        nodeAttributes.addCDATAAttribute("type", "node");
        
//        String realOrder = hasOrderableChildNodes ? "document" : order;
        
        XMLUtils.startElement(contentHandler, "node", nodeAttributes);
        
        // Properties
        _saxProperties(node);

        // Children
        _saxChildren(node, start, end, order);
        
        // Referers
        _saxReferers(node);
        
        XMLUtils.endElement(contentHandler, "node");
    }

    private void _saxProperties(Node node) throws SAXException, RepositoryException
    {
        Map<String, Property> jcrProperties = new TreeMap<>();
        Map<String, Property> otherProperties = new TreeMap<>();
        
        PropertyIterator itProperty = node.getProperties();
        while (itProperty.hasNext())
        {
            Property property = itProperty.nextProperty();
            String propertyName = property.getName();
            
            if (propertyName.startsWith("jcr:"))
            {
                jcrProperties.put(propertyName, property);
            }
            else
            {
                otherProperties.put(propertyName, property);
            }
        }
        
        for (Property property : jcrProperties.values())
        {
            _saxProperty(property, node);
        }
        
        for (Property property : otherProperties.values())
        {
            _saxProperty(property, node);
        }
        
        XMLUtils.createElement(contentHandler, "isLocked", String.valueOf(node.isLocked()));
        XMLUtils.createElement(contentHandler, "isCheckedOut", String.valueOf(node.isCheckedOut()));
        XMLUtils.createElement(contentHandler, "isNew", Boolean.toString(node.isNew()));
    }
    
    private void _saxProperty(Property property, Node parentNode) throws SAXException, RepositoryException
    {
        boolean isMultiple = property.getDefinition().isMultiple();
        AttributesImpl propertyAttributes = new AttributesImpl();
        boolean editable = !property.getDefinition().isProtected() && !parentNode.isLocked();

        propertyAttributes.addCDATAAttribute("name", property.getName());
        propertyAttributes.addCDATAAttribute("multiple", String.valueOf(isMultiple));
        propertyAttributes.addCDATAAttribute("type", PropertyType.nameFromValue(property.getType()));
        propertyAttributes.addCDATAAttribute("editable", String.valueOf(editable));
        propertyAttributes.addCDATAAttribute("new", Boolean.toString(property.isNew()));
        propertyAttributes.addCDATAAttribute("modified", Boolean.toString(property.isModified()));
        
        XMLUtils.startElement(contentHandler, "property", propertyAttributes);
        
        if (isMultiple)
        {
            Value[] values = property.getValues();
            
            for (int i = 0; i < values.length; i++)
            {
                _saxValue(values[i]);
            }
        }
        else
        {
            _saxValue(property.getValue());
        }
        
        XMLUtils.endElement(contentHandler, "property");
    }
    
    private void _saxChildren(Node node, long start, long end, String order) throws SAXException, RepositoryException
    {
        NodeIterator itNode = node.getNodes();
        
        long startIndex = Math.max(0, start);
        long endIndex = end >= 0 ? end : (itNode.getSize() - 1);
        
        long childCount = endIndex - startIndex + 1;
        
        if (childCount > NodeGroupHelper.MAX_CHILDREN_PER_NODE)
        {
            _saxGroups(node, startIndex, childCount);
        }
        else
        {
            if ("alpha".equals(order) || "reverseAlpha".equals(order))
            {
                Collection<Node> children = _extractNodes(itNode, startIndex, endIndex, "alpha".equals(order));
                for (Node child : children)
                {
                    _saxSubnode(child);
                }
            }
            else
            {
                itNode.skip(startIndex);
                for (long i = startIndex; i <= endIndex && itNode.hasNext(); i++)
                {
                    _saxSubnode(itNode.nextNode());
                }
            }
        }
    }
    
    private Collection<Node> _extractNodes(NodeIterator itNodes, long start, long end, boolean ascending)
    {
        List<Node> nodes = new ArrayList<>((int) itNodes.getSize());
        
        // Fill in the list with all elements.
        while (itNodes.hasNext())
        {
            nodes.add(itNodes.nextNode());
        }
        
        // Sort the collection.
        Collections.sort(nodes, ascending ? __ALPHA_NODE_COMPARATOR : __REVERSE_ALPHA_NODE_COMPARATOR);
        
        return nodes.subList((int) start, (int) end + 1);
    }
    
    /**
     * Generate node groups.
     * @param node The node to consider
     * @param startIndex The index to start the group
     * @param childCount The group size
     * @throws RepositoryException if an error occurred
     * @throws SAXException if an error occurred
     */
    private void _saxGroups(Node node, long startIndex, long childCount) throws RepositoryException, SAXException
    {
        // Compute the number of levels.
        long level = (long) Math.ceil(Math.log(childCount) / Math.log(NodeGroupHelper.MAX_CHILDREN_PER_NODE)) - 1;
        long childrenPerGroup = (long) Math.pow(NodeGroupHelper.MAX_CHILDREN_PER_NODE, level);
        long groupCount = (long) Math.ceil((double) childCount / (double) childrenPerGroup);
        
        // SAX virtual nodes (groups).
        for (long i = 0; i < groupCount; i++)
        {
            long groupStartIndex = startIndex + i * childrenPerGroup;
            long groupEndIndex = startIndex + Math.min((i + 1) * childrenPerGroup - 1 , childCount - 1);
            String name = String.format("%d...%d", groupStartIndex + 1, groupEndIndex + 1);
            
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("level", Long.toString(level));
            atts.addCDATAAttribute("index", Long.toString(i));
            atts.addCDATAAttribute("start", Long.toString(groupStartIndex));
            atts.addCDATAAttribute("end", Long.toString(groupEndIndex));
            atts.addCDATAAttribute("path", node.getPath());
            atts.addCDATAAttribute("name", name);
            atts.addCDATAAttribute("type", "group");
            XMLUtils.createElement(contentHandler, "group", atts);
        }
    }

    /**
     * Sax a sub node.
     * @param node the sub node to SAX.
     * @throws RepositoryException if an error occurred
     * @throws SAXException if an error occurred
     */
    private void _saxSubnode(Node node) throws RepositoryException, SAXException
    {
        AttributesImpl subNodeAttributes = new AttributesImpl();
        int index = node.getIndex();
        String name = node.getName();
        
        if (index != 1)
        {
            name = name + "[" + index + "]";
        }
        
        subNodeAttributes.addCDATAAttribute("parentPath", node.getParent().getPath());
        subNodeAttributes.addCDATAAttribute("name", name);
        subNodeAttributes.addCDATAAttribute("index", String.valueOf(index));
        subNodeAttributes.addCDATAAttribute("hasOrderableChildNodes", Boolean.toString(node.getPrimaryNodeType().hasOrderableChildNodes()));
        subNodeAttributes.addCDATAAttribute("isNew", Boolean.toString(node.isNew()));
        subNodeAttributes.addCDATAAttribute("isModified", Boolean.toString(node.isModified()));
        subNodeAttributes.addCDATAAttribute("type", "node");
        
        if (!node.hasNodes())
        {
            subNodeAttributes.addCDATAAttribute("noChild", "true");
        }
        
        XMLUtils.createElement(contentHandler, "node", subNodeAttributes);
    }

    private void _saxReferers(Node node) throws SAXException, RepositoryException
    {
        PropertyIterator itReference = node.getReferences();

        while (itReference.hasNext())
        {
            Property property = itReference.nextProperty();
            AttributesImpl referenceAttributes = new AttributesImpl();
            
            Node parent = property.getParent();
            
            referenceAttributes.addCDATAAttribute("path", parent.getPath());
            referenceAttributes.addCDATAAttribute("pathWithGroups", NodeGroupHelper.getPathWithGroups(parent));
            referenceAttributes.addCDATAAttribute("name", property.getName());
            
            XMLUtils.createElement(contentHandler, "referer", referenceAttributes);
        }
    }

    private void _saxValue(Value value) throws RepositoryException, SAXException
    {
        Attributes attrs = XMLUtils.EMPTY_ATTRIBUTES;
        String textValue = null;
        
        switch (value.getType())
        {
            case PropertyType.BINARY:
                textValue = "";
                break;
            case PropertyType.DATE:
                textValue = DateUtils.dateToString(value.getDate().getTime());
                break;
            case PropertyType.STRING:
            case PropertyType.PATH:
            case PropertyType.NAME:
            case PropertyType.LONG:
            case PropertyType.BOOLEAN:
            case PropertyType.REFERENCE:
            default:
                textValue = value.getString();
                break;
        }
        
        if (textValue != null)
        {
            XMLUtils.createElement(contentHandler, "value", attrs, textValue);
        }
    }
    
    /**
     * Compares two nodes on its names.
     */
    protected static class NodeNameComparator implements Comparator<Node>
    {
        
        /** Indicates if the comparators acts in ascending order. */
        protected boolean _ascending;
        
        /**
         * Constructs a node name comparator.
         */
        public NodeNameComparator()
        {
            this(true);
        }
        
        /**
         * Constructs a node name comparator, specifying ascending or descending order.
         * @param ascending true if we want to compare node names in ascending order, false otherwise.
         */
        public NodeNameComparator(boolean ascending)
        {
            this._ascending = ascending;
        }
        
        @Override
        public int compare(Node o1, Node o2)
        {
            try
            {
                if (_ascending)
                {
                    return o1.getName().compareTo(o2.getName());
                }
                else
                {
                    return o2.getName().compareTo(o1.getName());
                }
            }
            catch (RepositoryException e)
            {
                throw new RuntimeException("Impossible to get a node name.", e);
            }
        }
        
    }
    
}
