/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.jcr;

import java.io.IOException;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

/**
 * Generates the session state.
 */
public class SessionInformationGenerator extends AbstractRepositoryGenerator
{
    
    /** The node state tracker. */
    protected NodeStateTracker _nodeStateTracker;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _nodeStateTracker = (NodeStateTracker) serviceManager.lookup(NodeStateTracker.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String workspaceName = parameters.getParameter("workspace", "default");
            
        _getRepository(request, workspaceName);

        contentHandler.startDocument();
        
        try
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("hasPendingChanges", Boolean.toString(_session.hasPendingChanges()));
            XMLUtils.startElement(contentHandler, "session", attrs);
            
            generateNewNodes(workspaceName);
            
            XMLUtils.endElement(contentHandler, "session");
        }
        catch (RepositoryException e)
        {
            throw new ProcessingException(e);
        }
                
        contentHandler.endDocument();
    }
    
    /**
     * Generate the new nodes.
     * @param workspaceName the workspace name
     * @throws SAXException if an error occurred
     */
    protected void generateNewNodes(String workspaceName) throws SAXException
    {
        Set<String> newNodes = _nodeStateTracker.getNewNodes(workspaceName);
        
        XMLUtils.startElement(contentHandler, "new-nodes");
        
        for (String newNodePath : newNodes)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("path", newNodePath);
            XMLUtils.createElement(contentHandler, "node", attrs);
        }
        
        XMLUtils.endElement(contentHandler, "new-nodes");
    }
    
}
