/*
*  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.core.persistence.IterablePersistenceManager;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.version.InternalVersionManagerImpl;
import org.slf4j.Logger;

import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.ProgressionTracker;
import org.ametys.core.util.DateUtils;
import org.ametys.runtime.plugin.component.LogEnabled;

/**
 * Jackrabbit maintenance tasks implementations should extends this class.
 */
public abstract class AbstractMaintenanceTask implements LogEnabled
{
    /** ContainerProgressionTracker */
    protected ContainerProgressionTracker _progress;

    /** Logger for traces */
    protected Logger _logger;
    
    private boolean _isFinished;

    /** The repository config
     * Must be null when _repositoryContext is not null
     */
    private RepositoryConfig _repositoryConfig;
    
    /** The repository context
     * Must be null when _repositoryConfig is not null
     */
    private RepositoryContext _repositoryContext;
    
    /**
     * Indicate if the task need to be executed with an offline repository
     * @return true if the repository need to be offline
     */
    public boolean requiresOffline()
    {
        return true;
    }
    
    /**
     * Execute the task
     * @param repositoryInfo a pair providing either the repository config (left) or the repository context (right). Only one will be valued. The other one is null.
     * @throws RepositoryException If an error with the repository occurs while the task is executed.
     */
    public void execute(Pair<RepositoryConfig, RepositoryContext> repositoryInfo) throws RepositoryException 
    {
        _repositoryConfig = repositoryInfo.getLeft();
        _repositoryContext = repositoryInfo.getRight();

        long startTime = System.currentTimeMillis();
        
        setLogger();

        _logger.info("Executing task.");

        try
        {
            // task specific initialization.
            initialize();

            // Perform the task.
            apply();

            // Mark the task as finished.
            setFinished();
        }
        catch (Exception e)
        {
            setInErrorState(e);
            throw e;
        }
        finally
        {
            close();

            _logger.info("End of the task.");

            long elapsedTime = System.currentTimeMillis() - startTime;
            _logger.info("Done in " + _getFormattedDuration(elapsedTime));
        }
    }

    /**
     * Initialize the tasks.
     * This method can also create the {@link ProgressionTracker} object bounded to the task.
     * @throws RepositoryException If a repository exception
     */
    protected void initialize() throws RepositoryException
    {
        return;
    }

    /**
     * Apply the tasks (within the execute method()).
     * @throws RepositoryException If a repository exception
     */
    protected abstract void apply() throws RepositoryException;

    /**
     * Close the tasks
     */
    protected void close()
    {
        if (_repositoryConfig != null)
        {
            // We are in safe mode. So we need to shutdown the repository if we created it
            if (_repositoryContext != null && _repositoryContext.getRepository() != null)
            {
                _logger.info("Shutting down the repository created for maintenance operation");
                _repositoryContext.getRepository().shutdown();
            }
        }
        return;
    }
    
    /**
     * Set the tasks logger.
     */
    protected abstract void setLogger();
    
    @Override
    public void setLogger(Logger logger)
    {
        _logger = logger;
    }

    /**
     * Get progress information.
     * @return a map containing information on the progress status.
     */
    public Map<String, Object> getProgressInfo() 
    {
        if (_progress != null)
        {
            return _progress.toJSON();
        }
        else
        {
            return null;
        }
    }

    /**
     * Is the execution of the task finished?
     * @return true if finished
     */
    public boolean isFinished()
    {
        return _isFinished;
    }

    private synchronized void setFinished()
    {
        _isFinished = true;
    }

    private synchronized void setInErrorState(Exception e)
    {
        _logger.error(e.getLocalizedMessage(), e);

        _isFinished = true;
    }

    /**
     * Transforms millisecond to a formatted string representing a duration.
     * @param elapsedTime milleseconds corresponding to the duration.
     * @return Pretty formatted string.
     */
    protected String _getFormattedDuration(long elapsedTime)
    {
        String formatted = DateUtils.formatDuration(elapsedTime);
        return StringUtils.defaultIfEmpty(formatted, "0s");
    }
    
    /**
     * Get the repository config
     * @return the repository config
     */
    protected RepositoryConfig getRepositoryConfig()
    {
        if (_repositoryConfig != null)
        {
            return _repositoryConfig;
        }
        else
        {
            return _repositoryContext.getRepositoryConfig();
        }
    }
    
    /**
     * Create a repository context.
     * If a repository context was provided, or a repository was already created, then an error will be thrown
     * @return the repository context
     * @throws RepositoryException if an error occurred
     */
    protected RepositoryContext createRepositoryContext() throws RepositoryException
    {
        if (_repositoryConfig != null && _repositoryContext == null)
        {
            _logger.info("Creating repository for maintenance operation");
            _repositoryContext = RepositoryContext.create(_repositoryConfig);
            return _repositoryContext;
        }
        else
        {
            _logger.error("Trying to instanciate a new repository when a repository is already existing");
            throw new IllegalStateException("Trying to instanciate a new repository when a repository is already existing");
        }
    }
    
    /**
     * Get the current repository context if one was provided or already created. Or instantiate a new one from the provided config.
     * @return a repository context
     * @throws RepositoryException when an error occurred
     */
    protected RepositoryContext getOrCreateRepositoryContext() throws RepositoryException
    {
        if (_repositoryContext == null)
        {
            return createRepositoryContext();
        }
        else
        {
            return _repositoryContext;
        }
    }
    
    
    /**
     * Get the repository.
     * The repository can either be retrieved from a provided {@link RepositoryContext} or from a context created by {@link #createRepositoryContext()}, {@link #createRepository()} or {@link #getOrCreateRepository()}.
     * @return the repository
     */
    protected RepositoryImpl getRepository()
    {
        return _repositoryContext.getRepository();
    }
    
    /**
     * Create a repository. If there is already an existing repository or repository context, a error will be thrown
     * @return a repository
     * @throws RepositoryException when an error occurred
     */
    protected RepositoryImpl createRepository() throws RepositoryException
    {
        return createRepositoryContext().getRepository();
    }
    
    /**
     * Get the repository from a provided context. Or instantiate a new repository from the provided config
     * @return a repository
     * @throws RepositoryException when an error occurred
     */
    protected RepositoryImpl getOrCreateRepository() throws RepositoryException
    {
        return getOrCreateRepositoryContext().getRepository();
    }
    
    /**
     * Get the persistence manager from all workspace and the version manager from the repository context
     * If any of those persistence manager is not an {@link IterablePersistenceManager} null will be returned
     * @param context the repository context
     * @return the persistence managers
     * @throws RepositoryException if an error occurred
     */
    protected static List<IterablePersistenceManager> getAllPersistenceManager(RepositoryContext context) throws RepositoryException
    {
        // Workaround to get the list of the PersistenceManager
        List<IterablePersistenceManager> pmList = new ArrayList<>();
    
        // PM of version manager
        InternalVersionManagerImpl vm = context.getInternalVersionManager();
        if (vm.getPersistenceManager() instanceof IterablePersistenceManager versionPersistenceManager)
        {
            // For some reason, previous implementation return null if any PM was not iterable.
            // Kept the implementation the same.
            pmList.add(versionPersistenceManager);
        }
        else
        {
            return null;
        }
    
        // PMs of workspaces.
        String[] wspNames = context.getWorkspaceManager().getWorkspaceNames();
        for (String wspName : wspNames)
        {
            final PersistenceManager persistenceManager = context.getWorkspaceInfo(wspName).getPersistenceManager();
            if (persistenceManager instanceof IterablePersistenceManager iterablePersistenceManager)
            {
                pmList.add(iterablePersistenceManager);
            }
            else
            {
                // For some reason, previous implementation return null if any PM was not iterable.
                // Kept the implementation the same.
                return null;
            }
        }
        return pmList;
    }

}
