/*
*  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.util.List;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.persistence.IterablePersistenceManager;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.persistence.bundle.AbstractBundlePersistenceManager;
import org.apache.jackrabbit.core.persistence.check.ConsistencyCheckListener;
import org.apache.jackrabbit.core.persistence.check.ConsistencyReport;
import org.apache.jackrabbit.core.persistence.check.ReportItem;
import org.slf4j.LoggerFactory;

import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * ConsistencyCheckTask
 */
public class ConsistencyCheckTask extends AbstractMaintenanceTask implements ConsistencyCheckListener
{
    /** The JCR Session bound to this task. */
    protected Session _session;

    private List<IterablePersistenceManager> _pmList;

    @Override
    protected void initialize() throws RepositoryException
    {
        // Create the repository and log in the session.
        RepositoryContext repositoryContext = createRepositoryContext();
        _session = repositoryContext.getRepository().login(new SimpleCredentials("__MAINTENANCE_TASK__", "".toCharArray()));

        _pmList = getAllPersistenceManager(repositoryContext);

        // Initialize the task progress object.
        int count = 0; // number of item that will be scanned.

        try
        {
            for (IterablePersistenceManager pm : _pmList)
            {
                count += pm.getAllNodeIds(null, 0).size();
            }
            
            _progress = ProgressionTrackerFactory.createContainerProgressionTracker(new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_CONSISTENCYCHECK"), _logger);
            // count for the number of nodes, plus 1 for closing the task
            _progress.setSize(count + 1);
        }
        catch (Exception e)
        {
            _progress.setSize(0);
            _logger.error(e.getLocalizedMessage(), e);
        }
    }
    
    @Override
    protected void setLogger()
    {
        setLogger(LoggerFactory.getLogger(ConsistencyCheckTask.class));
    }

    @Override
    protected void apply() throws RepositoryException
    {
        for (PersistenceManager pm : _pmList)
        {
            // Do PM consistency check
            if (pm instanceof AbstractBundlePersistenceManager abpm)
            {
                // Perform the check
                // null -> all uuids, true -> recursive, false -> nofix, null, 
                // lost+found -> null, this -> listener 
                ConsistencyReport report = abpm.check(null, true, false, null, this);
                int numberOfNodesChecked = report.getNodeCount();
                
                _logger.info("Consistency check done for persistence manager : '" + pm.toString() + "' in " + (report.getElapsedTimeMs() / 1000f) + " s.");
                _logger.info(numberOfNodesChecked + " nodes were checked.");
                _logger.info(report.getItems().isEmpty() ? "No consistency problems were reported." : report.getItems().size() + " consistency problems were reported."); 
                
                _progress.increment(numberOfNodesChecked);
            }
        }
    }

    @Override
    protected void close()
    {
        if (_session != null)
        {
            _session.logout();
        }
        
        // We need to close the session before shuting the repository down
        super.close();
        
        if (_progress != null)
        {
            _progress.increment();
        }
    }

    // Listener methods 

    @Override
    public void startCheck(String id)
    {
        // Do Nothing
    }

    @Override
    public void report(ReportItem item)
    {
        _logger.warn(item.toString());
    }

    @Override
    public void error(String id, String message)
    {
        _logger.error("error during the consistency check -> id : [ " + id + "]\n" + message);
    }

    @Override
    public void info(String id, String message)
    {
        _logger.info("error during the consistency check -> id : [ " + id + "]\n" + message);
    }
}
