/*
*  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.sql.DataSource;

import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.api.management.MarkEventListener;
import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.data.DataIdentifier;
import org.apache.jackrabbit.core.data.DataStore;
import org.apache.jackrabbit.core.data.DataStoreException;
import org.apache.jackrabbit.core.data.FileDataStore;
import org.apache.jackrabbit.core.data.db.DbDataStore;
import org.apache.jackrabbit.core.data.db.DerbyDataStore;
import org.apache.jackrabbit.core.gc.GarbageCollector;
import org.apache.jackrabbit.core.persistence.IterablePersistenceManager;
import org.apache.jackrabbit.core.util.db.ConnectionFactory;
import org.apache.jackrabbit.core.util.db.ConnectionHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * DataStoreGarbageCollectorTask
 */
public class DataStoreGarbageCollectorTask extends AbstractMaintenanceTask implements MarkEventListener
{
    private static final int SYSTEM_GC_CALLS = 3;

    /** The JCR Session bound to this task. */
    protected Session _session;

    private GarbageCollector _garbageCollector;
    private List<IterablePersistenceManager> _pmList;

    /** Internal counter for scanned node by the GC */
    private int _scannedNodesCount;

    @Override
    protected void initialize() throws RepositoryException
    {
        // Create the repository and log in the session.
        RepositoryContext repositoryContext = createRepositoryContext();
        _session = repositoryContext.getRepository().login(new SimpleCredentials("__MAINTENANCE_TASK__", "".toCharArray()));

        // Create the GC Object
        this._garbageCollector = repositoryContext.getRepository().createDataStoreGarbageCollector();

        _pmList = getAllPersistenceManager(repositoryContext);

        // Initialize the task progress object.
        int count = 0; // number of item that will be scanned.

        try
        {
            for (IterablePersistenceManager pm : _pmList)
            {
                count += pm.getAllNodeIds(null, 0).size();
            }
            
            _progress = ProgressionTrackerFactory.createContainerProgressionTracker(new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_STARTGARBAGECOLLECTOR"), _logger);
            // When scan is finished, progress is set at 70%.
            SimpleProgressionTracker progressTrackerForScan = _progress.addSimpleStep("scan", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_MAINTENANCE_STARTGARBAGECOLLECTOR_SCAN_STEP_LABEL"), 7).withLoggerGranularity(10000);
            progressTrackerForScan.setSize(count);
            
            _progress.addSimpleStep("deletingunuseditems", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_MAINTENANCE_STARTGARBAGECOLLECTOR_DELETE_UNUSED_ITEMS_STEP_LABEL"), 2);
            SimpleProgressionTracker progressTrackerForFinishing = _progress.addSimpleStep("finishing", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_MAINTENANCE_STARTGARBAGECOLLECTOR_FINISH_STEP_LABEL"));
            progressTrackerForFinishing.setSize(2);
        }
        catch (Exception e)
        {
            _progress.setSize(0);
            _logger.error(e.getLocalizedMessage(), e);
        }
    }
    
    @Override
    protected void setLogger()
    {
        setLogger(LoggerFactory.getLogger(DataStoreGarbageCollectorTask.class));
    }

    @Override
    protected void apply() throws RepositoryException
    {
        // call System.gc() a few times before running the data store garbage collection. 
        // Please note System.gc() does not guarantee all objects are garbage collected.
        for (int i = 0; i < SYSTEM_GC_CALLS; i++)
        {
            System.gc();
        }

        // Log some information about the ds.
        long startSize = _reportDataStoreInfo(_garbageCollector.getDataStore());

        // Sleep
        if (_garbageCollector.getDataStore() instanceof FileDataStore)
        {
            // make sure the file is old (access time resolution is 2 seconds)
            try
            {
                Thread.sleep(2000);
            }
            catch (InterruptedException e)
            {
                _logger.error(e.getLocalizedMessage(), e);
                throw new RuntimeException(e); 
            }
        }

        // GC Management
        _garbageCollector.setMarkEventListener(this);
        _garbageCollector.setPersistenceManagerScan(true);
        // gc.setSleepBetweenNodes(0);

        // Run GC
        try
        {
            _scannedNodesCount = 0;
            _garbageCollector.mark();
            _logger.info(_scannedNodesCount + " nodes scanned.");

            int deleted = _garbageCollector.sweep();
            _logger.info(deleted + " unused items deleted.");
            
            ((SimpleProgressionTracker) _progress.getStep("deletingunuseditems")).increment();

            // Compressing derby DATASTORE table.
            if (_garbageCollector.getDataStore() instanceof DerbyDataStore)
            {
                _logger.info("Reclaiming unused space, this may take several minutes depending on the size of the data store.");
                _logger.info("Please be patient.");

                DbDataStore ds = (DbDataStore) _garbageCollector.getDataStore();
                DataSource dataSource = _getDataSource(ds);
                String table = ds.getTablePrefix() + ds.getSchemaObjectPrefix() + "DATASTORE";
                derbyCompressTable(dataSource, table, _logger);
            }
        }
        finally
        {
            _garbageCollector.close();
            ((SimpleProgressionTracker) _progress.getStep("finishing")).increment();
        }

        // Log some information about the ds.
        long finalSize = _reportDataStoreInfo(_garbageCollector.getDataStore());
        long freedSize = startSize - finalSize;
        _logger.info("Size of cleared data: " + FileUtils.byteCountToDisplaySize(freedSize) + "Ko");
        _logger.info("The total released space on your disk can be different depending on the type of the data store used by your repository.");
    }

    @Override
    protected void close()
    {
        if (_session != null)
        {
            _session.logout();
        }

        super.close();

        if (_progress != null)
        {
            ((SimpleProgressionTracker) _progress.getStep("finishing")).increment();
        }
    }

    private long _reportDataStoreInfo(DataStore ds) throws DataStoreException
    {
        long count = 0;
        long total = 0;
        Iterator<DataIdentifier> it = ds.getAllIdentifiers();
        while (it.hasNext())
        {
            count++;
            DataIdentifier id = it.next();
            total += ds.getRecord(id).getLength();
        }

        StringBuilder sb = new StringBuilder();
        sb.append("Datastore item count: ").append(count).append(" ");
        sb.append("[total size: ").append(FileUtils.byteCountToDisplaySize(total)).append("]");
        _logger.info(sb.toString());

        return total;
    }

    /**
     * Retrieve the data source of a data store.
     * @param ds the data store
     * @return the data source
     * @throws RepositoryException when an error occurred
     */
    protected DataSource _getDataSource(DbDataStore ds) throws RepositoryException
    {
        DataSource dataSource = null;

        try
        {
            ConnectionFactory cf = getRepositoryConfig().getConnectionFactory();
            if (ds.getDataSourceName() == null || "".equals(ds.getDataSourceName()))
            {
                dataSource = cf.getDataSource(ds.getDriver(), ds.getUrl(), ds.getUser(), ds.getPassword());
            }
            else
            {
                dataSource = cf.getDataSource(ds.getDataSourceName());
            }
        }
        catch (SQLException e)
        {
            _logger.error(e.getLocalizedMessage(), e);
            throw new RuntimeException(e);
        }
        catch (RepositoryException e)
        {
            _logger.warn("Failed to retrieve the data source from data store " + ds.getUrl(), e);
            throw e;
        }
        return dataSource;
    }
    
    /**
     * Reclaiming unused space. This is derby specific.
     * By default, Derby does not return unused space to the operating system
     * when updating or deleting data.
     * @param dataSource the data source to compress
     * @param table the table to compress
     * @param logger the logger to use for error
     */
    public static void derbyCompressTable(DataSource dataSource, String table, Logger logger)
    {
        ConnectionHelper conHelper = new ConnectionHelper(dataSource, false);
        String sql = "CALL SYSCS_UTIL.SYSCS_COMPRESS_TABLE(CURRENT SCHEMA, ?, 0)";

        try
        {
            conHelper.query(sql, table);
        }
        catch (SQLException e)
        {
            logger.error(e.getLocalizedMessage(), e);
            throw new RuntimeException(e);
        }
    }
    
    @Override
    public void beforeScanning(Node n) throws RepositoryException
    {
        _scannedNodesCount++;
        if (_progress != null)
        {
            ((SimpleProgressionTracker) _progress.getStep("scan")).increment();
        }
    }
}
