/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.workspaces.repository.maintenance.MaintenanceTaskManager.MaintenanceTaskType;

/**
 * Component to launch and follow the maintenance tasks on the repository
 */
public class MaintenanceTaskComponent extends AbstractLogEnabled implements Component, Serviceable
{
    /** The maintenance task manager */
    private MaintenanceTaskManager _taskManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _taskManager = (MaintenanceTaskManager) smanager.lookup(MaintenanceTaskManager.ROLE);
    }

    /**
     * Start a Maintenance Task
     * @param taskName the name of the task
     * @return a JSon object
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> startTask(String taskName)
    {
        Map<String, Object> result = new HashMap<>();

        try
        {
            boolean launched = _taskManager.launch(MaintenanceTaskType.valueOf(taskName));
            result.put("launched", launched);
        }
        catch (Exception e)
        {
            getLogger().error(e.getMessage(), e);
            result.put("launched", false);
        }
        
        result.put("running", _taskManager.isTaskRunning());
        
        return result;
    }

    /**
     * Check if a task is running or not
     * @param taskName The task name
     * @return true if the task is running
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> isRunning(String taskName)
    {
        Map<String, Object> result = new HashMap<>();

        String isRunning = Boolean.toString(_taskManager.isTaskRunning());
        result.put("running", isRunning);
        
        return result;
    }

    /**
     * Get the information of a running task
     * @param taskName the task name
     * @return the information
     */
    @Callable(rights = "REPOSITORY_Rights_Access", context = "/repository")
    public Map<String, Object> getInformation(String taskName)
    {
        Map<String, Object> result = new HashMap<>();

        // Is task finished ?
        boolean taskRunning = _taskManager.isTaskRunning();
        result.put("isRunning", taskRunning);
        if (taskRunning)
        {
            result.put("taskName", StringUtils.defaultString(_taskManager.getRunningTaskType()));
        }
        else
        {
            result.put("lastTaskName", StringUtils.defaultString(_taskManager.getLastRunningTaskType()));
        }
        
        // progress to json
        result.put("progress", _taskManager.getProgressInfo());

        // repository state to json
        result.put("respositoryState", _taskManager.isRepositoryStarted());

        return result;
    }
}
