/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.io.File;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;

import javax.jcr.RepositoryException;
import javax.jcr.UnsupportedRepositoryOperationException;
import javax.servlet.ServletException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.environment.Context;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.jackrabbit.core.RepositoryContext;
import org.apache.jackrabbit.core.config.ConfigurationException;
import org.apache.jackrabbit.core.config.RepositoryConfig;

import org.ametys.core.util.SystemStatus;
import org.ametys.plugins.repositoryapp.RepositoryProvider;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * The MaintenanceTaskManager Component
 */
public class MaintenanceTaskManager extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The avalon role. */
    public static final String ROLE = MaintenanceTaskManager.class.getName();
    
    /** The repository unavailable system status */
    public static final String REPOSITORY_UNAVAILABLE = "REPOSITORY_UNAVAILABLE";
    
    /** The maintenance task running system status */
    public static final String MAINTENANCE_TASK_RUNNING = "MAINTENANCE_TASK_RUNNING";
    
    /** The maintenance task ended system status */
    public static final String MAINTENANCE_TASK_ENDED = "MAINTENANCE_TASK_ENDED";
    
    /** The maintenance running task */
    protected static AbstractMaintenanceTask _runningTask;
    
    /** The maintenance running task */
    protected static MaintenanceTaskType _lastRunningTaskType;
    
    /** The maintenance running task type */
    protected MaintenanceTaskType _runningTaskType;

    /** System status provider */
    protected SystemStatus _systemStatus;
    
    /** The repository provider. */
    protected RepositoryProvider _repositoryProvider;
    
    private final ExecutorService _executor = Executors.newFixedThreadPool(1);
    
    private FutureTask<Void> _future;

    private boolean _repositoryShutdown;

    private Context _context;
    
    /** Task types */
    public enum MaintenanceTaskType
    {
        /** Remove unused history */
        REMOVE_UNUSED_HISTORY,
        /** data store GC */
        DATA_STORE_GARBAGE_COLLECTOR,
        /** reindexing task */
        REINDEXING,
        /** consistency check */
        CONSISTENCY_CHECK,
        /** remove inconsistent references */
        REMOVE_INCOHERENT_REFERENCES,
        /** reclaim unused space in the derby data store */
        RECLAIM_UNUSED_SPACE;
    }

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        if (manager.hasService(RepositoryProvider.ROLE))
        {
            // In safe-mode, the repository provider can not be retrieved
            _repositoryProvider = (RepositoryProvider) manager.lookup(RepositoryProvider.ROLE);
        }
        _systemStatus = (SystemStatus) manager.lookup(SystemStatus.ROLE);
    }

    /**
     * Launch a maintenance task
     * @param type The maintenance task type
     * @return true if a task has been launched
     * @throws Exception when an error occurred
     */
    public boolean launch(MaintenanceTaskType type) throws Exception
    {
        boolean launched = false;
        if (!isTaskRunning())
        {
            try
            {
                run(type);
                launched = true;
            }
            catch (ConfigurationException e)
            {
                getLogger().error("Unable to get the configuration of the repository.", e);
                throw e;
            }
            catch (RepositoryException e)
            {
                getLogger().error("Unable to launch the task.", e);
                throw e;
            }
        }
        return launched;
    }

    private void run(final MaintenanceTaskType type) throws ConfigurationException, RepositoryException
    {
        _runningTaskType = type;
        _runningTask = _createTask(type);
        
        // add system status
        if (_runningTask.requiresOffline())
        {
            _systemStatus.addStatus(REPOSITORY_UNAVAILABLE);
        }
        _systemStatus.addStatus(MAINTENANCE_TASK_RUNNING);
        _systemStatus.removeStatus(MAINTENANCE_TASK_ENDED);
        
        final Logger logger = getLogger();
        
        
        Pair<RepositoryConfig, RepositoryContext> repositoryInfo = _getRepositoryInfo();
        
        _future = new FutureTask<>(new Callable<Void>() 
        {
            public Void call() throws ServletException
            {
                try
                {
                    _runningTask.execute(repositoryInfo);
                }
                catch (RepositoryException e)
                {
                    logger.error(e.getMessage(), e);
                }
                finally
                {
                    // Currently repository cannot be restarted, the Ametys application have to be reloaded.
                    // So the REPOSITORY_UNAVAILABLE system status cannot be removed
                    // _systemStatus.removeStatus(REPOSITORY_UNAVAILABLE);
                    _systemStatus.removeStatus(MAINTENANCE_TASK_RUNNING);
                    _systemStatus.addStatus(MAINTENANCE_TASK_ENDED);
                    _runningTaskType = null;
                }

                return null;
            }
        });
        
        _executor.execute(_future);
    }

    /**
     * Retrieve the repository config if no repository is available or the repository context of the available repository
     * @return Either the config or the context depending if the repository is available. The other one will be null.
     * @throws ConfigurationException if an error occurred while generating the config
     * @throws UnsupportedRepositoryOperationException if the repository doesn't support retrieving the context
     */
    protected Pair<RepositoryConfig, RepositoryContext> _getRepositoryInfo() throws ConfigurationException, UnsupportedRepositoryOperationException
    {
        final RepositoryConfig repositoryConfig;
        final RepositoryContext repositoryContext;
        
        if (_repositoryProvider == null)
        {
            repositoryConfig = _createRepositoryConfig();
            repositoryContext = null;
        }
        else
        {
            _runningTaskType = null;
            throw new UnsupportedRepositoryOperationException("A maintenance task can not be run on an online repository instance of " + _repositoryProvider.getRepository().getClass().getName());
        }
        return Pair.of(repositoryConfig, repositoryContext);
    }
    
    /**
     * Create a repository config
     * @return the repository config
     * @throws ConfigurationException if an error occurred
     */
    protected RepositoryConfig _createRepositoryConfig() throws ConfigurationException
    {
        String config = _context.getRealPath("/WEB-INF/param/repository.xml");
        
        File homeFile = new File(AmetysHomeHelper.getAmetysHomeData(), "repository");
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Creating JCR Repository config at: " + homeFile.getAbsolutePath());
        }
        
        return RepositoryConfig.create(config, homeFile.getAbsolutePath());
    }
    
    /**
     * Initialize the tasks.
     * @param type MaintenanceTaskType
     * @return the task.
     */
    protected AbstractMaintenanceTask _createTask(MaintenanceTaskType type)
    {
        switch (type)
        {
            case REMOVE_UNUSED_HISTORY: //online + renforce pour
                return new RemoveUnusedHistoryTask();
            case DATA_STORE_GARBAGE_COLLECTOR:
                return new DataStoreGarbageCollectorTask();
            case REINDEXING: // offline
                return new ReindexingTask();
            case CONSISTENCY_CHECK: // ??
                return new ConsistencyCheckTask();
            default:
                throw new IllegalArgumentException("This type of maintenance task is not allowed : " + type);
        }
    }

    /**
     * Indicates if a maintenance task is running.
     * @return true if a task is running.
     */
    public boolean isTaskRunning()
    {
        return _runningTaskType != null;
    }

    /**
     * Get progress information.
     * @return a map containing information on the progress status.
     */
    public Map<String, Object> getProgressInfo()
    {
        if (_runningTask != null)
        {
            return _runningTask.getProgressInfo();
        }

        return null;
    }

    /**
     * Retrieves the type of the running task
     * @return the type of the running task or null.
     */
    public String getRunningTaskType()
    {
        if (_runningTaskType != null)
        {
            return _runningTaskType.name();
        }

        return null;
    }
    
    /**
     * Retrieves the type of the previous running task, if no task is currently running.
     * Otherwise, the current task type will be return.
     * @return the type or null
     */
    public String getLastRunningTaskType()
    {
        if (_lastRunningTaskType != null)
        {
            return _lastRunningTaskType.name();
        }
        
        return null;
    }

    /**
     * Indicates is the Ametys repository is started or has been shut down.
     * @return true is the repository is started
     */
    public boolean isRepositoryStarted()
    {
        return !_repositoryShutdown;
    }
}
