/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.io.File;
import java.io.IOException;

import javax.jcr.RepositoryException;

import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.slf4j.LoggerFactory;

import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * ReindexingTask
 */
public class ReindexingTask extends AbstractMaintenanceTask
{
    private static final String _INDEX_FOLDER_RELATIVE_PATH = "repository" + File.separator + "index";
    private static final String _WORKSPACES_FOLDER_RELATIVE_PATH = "workspaces";

    private String[] _workspaceFolders;

    @Override
    protected void initialize() throws RepositoryException
    {
        File file = new File(getRepositoryConfig().getHomeDir() + File.separator + _WORKSPACES_FOLDER_RELATIVE_PATH);
        _workspaceFolders = file.list((File f, String name) -> f.isDirectory());
        
        // Initialize the task progress object.
        _progress = ProgressionTrackerFactory.createContainerProgressionTracker(new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_REINDEXING"), _logger);
        
        // deleting workspace index folders + repository root index folder = 60 %
        SimpleProgressionTracker progressTrackerForScan = _progress.addSimpleStep("deletefolders", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_MAINTENANCE_REINDEXING_DELETE_FOLDERS_STEP_LABEL"), 6);
        progressTrackerForScan.setSize(_workspaceFolders.length + 1);
        
        // reindexing = 39 %
        _progress.addSimpleStep("reindexing", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_REINDEXING"), 3.9);
        // shutting down = 1 %
        _progress.addSimpleStep("finishing", new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_MAINTENANCE_REINDEXING_SHUTING_DOWN_REPOSITORY_STEP_LABEL"), 0.1);
    }

    @Override
    protected void setLogger()
    {
        setLogger(LoggerFactory.getLogger(ReindexingTask.class));
    }
    
    @Override
    protected void apply() throws RepositoryException
    {
        // Deleting repository root index folder
        try
        {
            deleteIndexFolder(_INDEX_FOLDER_RELATIVE_PATH);
            _logger.info("Successfully deleted root repository index folder");
        }
        catch (IOException e)
        {
            _logger.error(e.getLocalizedMessage(), e);
        }

        // Deleting workspace index folders
        for (String folder : _workspaceFolders)
        {
            try
            {
                deleteIndexFolder(_WORKSPACES_FOLDER_RELATIVE_PATH  + File.separator + folder + File.separator + "index");
                _logger.info("Successfully deleted index folder of workspace '" + folder + "'");
            }
            catch (IOException e)
            {
                _logger.error(e.getLocalizedMessage(), e);
            }
        }

        // Creates a repository instance to perform the re-indexing process.
        _logger.info("Starting repository to launch the re-indexing process");
        RepositoryImpl repository = createRepository();
        _logger.info("Repository restarted successfully, reindexing process has ended.");
        ((SimpleProgressionTracker) _progress.getStep("reindexing")).increment();

        // logout
        repository.shutdown();
        repository = null;
        ((SimpleProgressionTracker) _progress.getStep("finishing")).increment();
    }

    private void deleteIndexFolder(String relativePath) throws IOException
    {
        File dir = new File(getRepositoryConfig().getHomeDir() + File.separator + relativePath);
        FileUtils.deleteDirectory(dir);
        if (_progress != null)
        {
            ((SimpleProgressionTracker) _progress.getStep("deletefolders")).increment();
        }
    }
}
