/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.Credentials;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.query.Query;
import javax.jcr.version.VersionHistory;

import org.apache.jackrabbit.JcrConstants;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.slf4j.LoggerFactory;

import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.plugins.repositoryapp.jcr.VersionHistoryHelper;
import org.ametys.plugins.repositoryapp.jcr.VersionHistoryHelper.RemoveHistoryResult;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Remove unused object from jcr history
 */
public class RemoveUnusedHistoryTask extends AbstractMaintenanceTask
{
    /** The JackRabbit RepositoryImpl */
    protected RepositoryImpl _repository;

    @Override
    public boolean requiresOffline()
    {
        return false;
    }
    
    @Override
    protected void setLogger()
    {
        setLogger(LoggerFactory.getLogger(RemoveUnusedHistoryTask.class));
    }

    @Override
    protected void apply() throws RepositoryException
    {
        Map<String, Session> sessions = null;
        Credentials credentials = new SimpleCredentials("ametys", new char[]{});
       
        try
        {
            _repository = getOrCreateRepository();

            String[] accessibleWorkspaceNames = _getWorkspaces(credentials);
            sessions = _loadSessions(accessibleWorkspaceNames);
            
            @SuppressWarnings("deprecation")
            Query q = sessions.get("default").getWorkspace().getQueryManager().createQuery("//element(*, " + JcrConstants.NT_VERSIONHISTORY + ")", Query.XPATH);
            NodeIterator nodes = q.execute().getNodes();
            long size = nodes.getSize();
            
            _logger.info("Found " + size + " total version histories");
            
            _progress = ProgressionTrackerFactory.createContainerProgressionTracker(new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_STARTREMOVEUNUSEDHISTORY"), _logger);
            _progress.setSize(size);
            
            _removeUnusedHistory(sessions, nodes, size);
        }
        catch (RepositoryException e)
        {
            if (_progress != null)
            {
                _progress.setSize(0);
            }
            
            throw e;
        }
        catch (Exception e)
        {
            if (_progress != null)
            {
                _progress.setSize(0);
            }
            
            throw new RuntimeException(e);
        }
        finally
        {
            _logout(sessions);
        }
    }
    
    private void _logout(Map<String, Session> sessions)
    {
        if (sessions != null)
        {
            for (Session session : sessions.values())
            {
                session.logout();
            }
        }
    }
    
    private String[] _getWorkspaces(Credentials credentials) throws RepositoryException
    {
        Session session = null;
        try
        {
            session = _repository.login(credentials, "default");
            return session.getWorkspace().getAccessibleWorkspaceNames();
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    private Map<String, Session> _loadSessions(String[] accessibleWorkspaceNames) throws RepositoryException
    {
        Map<String, Session> sessions = new HashMap<>();
        Credentials credentials = new SimpleCredentials("ametys", new char[]{});
       
        for (String workspaceName: accessibleWorkspaceNames)
        {
            sessions.put(workspaceName, _repository.login(credentials, workspaceName));
        }
        
        return sessions;
    }
    
    private void _removeUnusedHistory(Map<String, Session> sessions, NodeIterator nodes, long size) throws RepositoryException
    {
        int errors = 0;
        int done = 0;
        int empty = 0;
        int fixed = 0;
        int count = 0;

        while (nodes.hasNext())
        {
            RemoveHistoryResult result = VersionHistoryHelper.removeUnusedHistory(sessions, (VersionHistory) nodes.next(), _logger);
            switch (result)
            {
                case ERROR:
                    errors++;
                    break;
                case FIXED:
                    fixed++;
                    break;
                case REMOVED:
                    done++;
                    break;
                case UNFIXABLE:
                    empty++;
                    break;
                case REFERENCED:
                default:
                    // nothing to do
                    break;
            }
            
            count++;

            if (count % 500 == 0)
            {
                _logger.info("{}/{} version histories handled, {} removed, {} fixed, {} unfixable, {} unconsistents", count, size, done, fixed, empty, errors);
            }
            
            _progress.increment();
        }

        _logger.info("{} normal history nodes removed\n"
                   + "{} empty history nodes fixed and removed\n"
                   + "{} empty history nodes that are unfixable\n"
                   + "{} unconsistents history nodes that cannot be removed",
                     done, fixed, empty, errors);
    }
}
