/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.repository.maintenance.ui;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.repositoryapp.RepositoryProvider;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for maintenance tasks. Checks if it is a JNDI repository
 */
public class MaintenanceTaskClientSideElement extends StaticClientSideElement
{
    private RepositoryProvider _repositoryProvider;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        if (smanager.hasService(RepositoryProvider.ROLE))
        {
            // In safe-mode, the repository provider can not be retrieved
            _repositoryProvider = (RepositoryProvider) smanager.lookup(RepositoryProvider.ROLE);
        }
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> clonedScripts = new ArrayList<>();
        
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        for (Script script : scripts)
        {
            Script clonedScript = new Script(script);
            
            I18nizableText unavailableDescription = getUnavailableDescription();
            if (unavailableDescription != null)
            {
                clonedScript.getParameters().put("unavailable", true);
                clonedScript.getParameters().put("unavailable-description", unavailableDescription);
            }
            
            clonedScripts.add(clonedScript);
        }
        
        return clonedScripts;
    }
    
    /**
     * Get the i18n description if the task is unavailable
     * @return the i18n description or null if the task is available
     */
    protected I18nizableText getUnavailableDescription ()
    {
        if (isJndi())
        {
            return new I18nizableText("plugin.repositoryapp", "PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_TASK_JNDI_DESC");
        }
        return null;
    }
    
    /**
     * Determines if the repository is obtained through JNDI
     * @return true if the repository is obtained through JNDI
     */
    protected boolean isJndi ()
    {
        return _repositoryProvider.isJndi();
    }
}

