/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class handles the access to AmetysObjects.
 * See {@link Ametys.plugins.repository.AmetysObject}.
 */
Ext.define('Ametys.plugins.repository.AmetysObjectDao',
{
    singleton: true,
    
    /**
     * Retrieve an AmetysObject by its id
     * @param {String} id The AmetysObject id. Cannot be null.
     * @param {Function} callback The callback function called when the AmetysObject is retrieved. Can be null for synchronous mode (not recommended!). Parameters are
     * @param {Ametys.plugins.repository.AmetysObject} callback.object The AmetysObject retrieved. Can be null if the object does not exist.
     */
    getAmetysObject: function(id, callback)
    {
        if (Ext.isEmpty(id))
        {
            callback(null);
            return;
        }
        this._sendRequest([id], Ext.bind(this._getAmetysObjectCb, this, [callback], 1));
    },
    
    /**
     * @private
     * Callback function called after #getAmetysObject is processed
     * @param {Ametys.plugins.repository.AmetysObject[]} objects The retrieved objects.
     * @param {Function} callback The callback function called 
     */
    _getAmetysObjectCb: function(objects, callback)
    {
        callback((objects.length == 0) ? null : objects[0]);
    },
    
    /**
     * Retrieve AmetysObjects by their ids
     * @param {String[]} ids The AmetysObject ids. Cannot be null.
     * @param {Function} callback The callback function called when the AmetysObject is retrieved. Can be null for synchronous mode (not recommended!). Parameters are
     * @param {Ametys.plugins.repository.AmetysObject} callback.object The AmetysObject retrieved. Can be null if object does not exist.
     */
    getAmetysObjects: function(ids, callback)
    {
        if (Ext.isEmpty(ids))
        {
            callback([]);
            return;
        }
        this._sendRequest(ids, Ext.bind(this._getAmetysObjectsCb, this, [callback], 1));
    },
    
    /**
     * @private
     * Callback function called after #getAmetysObjects is processed
     * @param {Ametys.plugins.repository.AmetysObject[]} objects The retrieved objects
     * @param {Function} callback The callback function called 
     */
    _getAmetysObjectsCb: function(objects, callback)
    {
        callback(objects);
    },
    
    /**
     * @private
     * Send request to server to retrieved objects
     * @param {String[]} ids The id of objects to retrieve
     * @param {Function} callback The callback function to be called after server process
     */
    _sendRequest: function(ids, callback)
    {
        Ametys.data.ServerComm.callMethod({
            role: 'org.ametys.plugins.repository.workspace.AmetysObjectDao', 
            methodName: 'getAmetysObjects',
            parameters: [ids],
            callback: {
                handler: this._sendRequestCb,
                scope: this,
                arguments: {callback: callback}
            },
            errorMessage: "{{i18n plugin.repositoryapp:PLUGINS_REPOSITORYAPP_REPOSITORY_DAO_GET_NODES_ERROR}}"
        });
    },
    
    /**
     * @private
     * Callback function called after #_sendRequest is processed
     * @param {Object} result The object retrieval result (found and not found). 
     * @param {Object} arguments The callback arguments.
     */
    _sendRequestCb: function(result, arguments)
    {
        var objects = [];
        for (var i=0; i < result.objects.length; i++)
        {
            objects.push(Ext.create('Ametys.plugins.repository.AmetysObject', result.objects[i]));
        }
        
        var objectsNotFound = result.notFound;
        if (objectsNotFound.length > 0)
        {
            Ametys.log.ErrorDialog.display({
                title: "{{i18n plugin.repositoryapp:PLUGINS_REPOSITORYAPP_REPOSITORY_DAO_NODES_NOT_FOUND_TITLE}}", 
                text: "{{i18n plugin.repositoryapp:PLUGINS_REPOSITORYAPP_REPOSITORY_DAO_NODES_NOT_FOUND_MESSAGE}}",
                details: "{{i18n plugin.repositoryapp:PLUGINS_REPOSITORYAPP_REPOSITORY_DAO_NODES_NOT_FOUND_DETAILS}}\n" + objectsNotFound.join('\n'),
                category: 'Ametys.plugins.repository.AmetysObjectDao'
            });
        }
        
        if (typeof arguments.callback == 'function')
        {
            arguments.callback(objects);
        }
    }
    
});
