/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Simple model representing an AmetysObject as a search result.
 */
Ext.define('Ametys.plugins.repository.tool.AmetysViewTool.SearchAmetysObject', {
    extend: 'Ext.data.TreeModel',
    fields: [
        {name: 'id', mapping: '@id'},
        {name: 'text', mapping: '@path'},
        {name: 'path', mapping: '@path'},
        {name: 'name', mapping: '@name'},
        {name: 'iconCls', defaultValue: 'a-tree-glyph ametysicon-ametys'},
        {name: 'leaf', defaultValue: true}
    ]
});

/**
 * This tool displays the Ametys logic repository view in a tree.
 * @private
 * 
 * FIXME Bug when the tool has never been focused (its tab is inactive) and tries to select a node (to reflect another selection). 
 * See https://www.sencha.com/forum/showthread.php?302708-Select-a-node-of-a-not-rendered-tree
 */
Ext.define('Ametys.plugins.repository.tool.AmetysViewTool',
{
    extend: 'Ametys.repository.tool.JcrViewTool',
    
    constructor: function(config)
    {
        this.callParent(arguments);
        
        this.messageTargetType = Ametys.message.MessageTarget.AMETYS_OBJECT;
    },
    
    _createTree: function ()
    {
        return Ext.create('Ametys.plugins.repository.tree.AmetysTreePanel', {
            flex: 1,
            split: true
        });
    },
    
    _createSearchTree: function ()
    {
        return Ext.create('Ametys.repository.tree.SearchTreePanel', {
            maxHeight: 200,
            split: true,
            searchUrl: Ametys.getPluginDirectPrefix('repository') + '/repository/logic-query',
            modelName: 'Ametys.plugins.repository.tool.AmetysViewTool.SearchAmetysObject'
        });
    },
    
    getMessageTargetConfiguration: function (record)
    {
        if (record == null)
        {
            // Empty selection
            return null;
        }
        else
        {
            return {
                id: this.messageTargetType,
                parameters: {
                    ids: [record.get('id')],
                    workspaceName: this._tree.getCurrentWorkspaceName()
                }
            };
        }
    },
    
    setInDirtyState: function (node)
    {
        // Not supported
    },
    
    _onMessageCreated: function(message)
    {
        this.callParent(arguments);
        
        var target = message.getTarget(Ametys.message.MessageTarget.REPOSITORY_NODE);
        if (target != null && target.getParameters().workspaceName == this._tree.getCurrentWorkspaceName())
        {
            var jcrPath = target.getParameters().path;
            var parentPath = jcrPath.substring(0, jcrPath.lastIndexOf('/'));
            
            var parentNode = this._tree.getNodeByJcrPath(parentPath);
            if (parentNode != null)
            {
                this._tree.getStore().load({node: parentNode});
            }
        }
    },
    
    _onMessageDeleted: function(message)
    {
        this.callParent(arguments);
        
        var target = message.getTarget(Ametys.message.MessageTarget.REPOSITORY_NODE);
        if (target != null)
        {
            var jcrPath = target.getParameters().path;
            var node = this._tree.getNodeByJcrPath(jcrPath);
            
            if (node != null)
            {
                var parentNode = node.parentNode;
                this._tree.getSelectionModel().select([parentNode]);
                node.remove();
            }
        }
    },
    
    openNode: function(view, node)
    {
        var workspaceName = Ametys.repository.RepositoryApp.getCurrentWorkspace() || this._tree.getCurrentWorkspaceName();
        Ametys.plugins.repository.tool.AmetysObjectMetadataTool.open(node.getId(), {workspaceName: workspaceName});
    }
});
